
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { DatePickerWithRange } from '@/components/ui/date-range-picker';
import { mockSellers, mockParcels } from '@/lib/data';
import { addDays, format } from 'date-fns';
import type { DateRange } from 'react-day-picker';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow, TableFooter as ShadTableFooter } from '@/components/ui/table';
import { Loader2, Download, Truck, Banknote, Receipt, WalletCards } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

type SettlementReport = {
    merchantId: string;
    merchantName: string;
    dateRange: DateRange;
    transactions: (Parcel & { fees: number; net: number })[];
    summary: {
        totalOrders: number;
        totalCod: number;
        totalFees: number;
        netPayable: number;
    };
};

export default function FinanceMerchantSettlementPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const [selectedMerchant, setSelectedMerchant] = useState<string>('');
  const [dateRange, setDateRange] = useState<DateRange | undefined>({
    from: addDays(new Date(), -7),
    to: new Date(),
  });
  const [isLoading, setIsLoading] = useState(false);
  const [report, setReport] = useState<SettlementReport | null>(null);
  const { toast } = useToast();

  const handleGenerate = () => {
    if (!selectedMerchant || !dateRange?.from || !dateRange?.to) {
        toast({ variant: 'destructive', title: 'Missing Information', description: 'Please select a merchant and a date range.' });
        return;
    }
    
    setIsLoading(true);
    setTimeout(() => {
        const merchantParcels = mockParcels.filter(p => 
            p.sellerId === selectedMerchant &&
            (p.status === 'Delivered' || p.status === 'Returned') &&
            new Date(p.bookingDate) >= dateRange.from! &&
            new Date(p.bookingDate) <= dateRange.to!
        );

        const transactions = merchantParcels.map(p => {
            const fees = p.status === 'Delivered' ? 150 : 75; // Mock fees
            const net = p.status === 'Delivered' ? p.codAmount - fees : -fees;
            return { ...p, fees, net };
        });

        const summary = transactions.reduce((acc, tx) => {
            acc.totalOrders++;
            if (tx.status === 'Delivered') {
                acc.totalCod += tx.codAmount;
            }
            acc.totalFees += tx.fees;
            acc.netPayable += tx.net;
            return acc;
        }, { totalOrders: 0, totalCod: 0, totalFees: 0, netPayable: 0 });

        setReport({
            merchantId: selectedMerchant,
            merchantName: mockSellers.find(s => s.id === selectedMerchant)?.name || '',
            dateRange,
            transactions,
            summary,
        });

        setIsLoading(false);
    }, 1000);
  };
  
  const MetricCard = ({ title, value, icon: Icon, isCurrency = true }: { title: string, value: number, icon: React.ElementType, isCurrency?: boolean }) => (
    <Card>
      <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
        <CardTitle className="text-sm font-medium">{title}</CardTitle>
        <Icon className="h-4 w-4 text-muted-foreground" />
      </CardHeader>
      <CardContent>
        <div className="text-2xl font-bold">
            {isCurrency ? `PKR ${value.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 })}` : value}
        </div>
      </CardContent>
    </Card>
  );

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Merchant Settlement Report</CardTitle>
              <CardDescription>
                Generate and view settlement reports for merchants based on a selected period.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <div className="p-4 border rounded-lg bg-muted/50 space-y-4">
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 items-end">
                        <div>
                            <label className="text-sm font-medium">Select Merchant</label>
                            <Select value={selectedMerchant} onValueChange={setSelectedMerchant}>
                                <SelectTrigger>
                                    <SelectValue placeholder="Select a merchant..." />
                                </SelectTrigger>
                                <SelectContent>
                                    {mockSellers.map(seller => (
                                        <SelectItem key={seller.id} value={seller.id}>{seller.name}</SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="md:col-span-1">
                            <label className="text-sm font-medium">Select Date Range</label>
                            <DatePickerWithRange date={dateRange} setDate={setDateRange} />
                        </div>
                        <Button onClick={handleGenerate} disabled={isLoading} className="w-full md:w-auto">
                            {isLoading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                            {isLoading ? 'Generating...' : 'Generate Report'}
                        </Button>
                    </div>
                </div>
            </CardContent>
          </Card>

          {report && (
                <Card className="animate-in fade-in-50">
                    <CardHeader>
                        <div className="flex justify-between items-start">
                            <div>
                                <h3 className="text-xl font-semibold">Settlement for {report.merchantName}</h3>
                                <p className="text-sm text-muted-foreground">
                                    Period: {report.dateRange?.from?.toLocaleDateString()} - {report.dateRange?.to?.toLocaleDateString()}
                                </p>
                            </div>
                            <Button variant="outline"><Download className="mr-2 h-4 w-4" /> Export PDF</Button>
                        </div>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                            <MetricCard title="Total Orders" value={report.summary.totalOrders} icon={Truck} isCurrency={false} />
                            <MetricCard title="Net COD Amount" value={report.summary.totalCod} icon={Banknote} />
                            <MetricCard title="Total Fees & Charges" value={report.summary.totalFees} icon={Receipt} />
                            <MetricCard title="Net Payable" value={report.summary.netPayable} icon={WalletCards} />
                        </div>
                        
                         <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Tracking #</TableHead>
                                    <TableHead>Status</TableHead>
                                    <TableHead className="text-right">COD Amount</TableHead>
                                    <TableHead className="text-right">Fees</TableHead>
                                    <TableHead className="text-right">Net Amount</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {report.transactions.length === 0 ? (
                                     <TableRow>
                                        <TableCell colSpan={5} className="text-center h-24">No transactions found for this period.</TableCell>
                                    </TableRow>
                                ) : report.transactions.map(tx => (
                                    <TableRow key={tx.id}>
                                        <TableCell className="font-medium">{tx.id}</TableCell>
                                        <TableCell>{tx.status}</TableCell>
                                        <TableCell className="text-right font-mono">
                                            {tx.status === 'Delivered' ? tx.codAmount.toFixed(2) : '0.00'}
                                        </TableCell>
                                        <TableCell className="text-right font-mono text-red-600">-{tx.fees.toFixed(2)}</TableCell>
                                        <TableCell className="text-right font-mono font-semibold">{tx.net.toFixed(2)}</TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                            <ShadTableFooter>
                                <TableRow>
                                    <TableCell colSpan={4} className="font-bold text-right">Net Payable to Merchant</TableCell>
                                    <TableCell className="text-right font-bold text-lg">
                                        PKR {report.summary.netPayable.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
                                    </TableCell>
                                </TableRow>
                            </ShadTableFooter>
                        </Table>
                    </CardContent>
                </Card>
            )}

        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}

