
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Seller } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { DatePickerWithRange } from '@/components/ui/date-range-picker';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { ShieldQuestion, Search, MoreVertical, CheckCircle, Hourglass, FileText, MessageSquarePlus } from 'lucide-react';
import { mockSellers } from '@/lib/data';
import { addDays, format } from 'date-fns';
import type { DateRange } from 'react-day-picker';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';

type DisputeStatus = 'Open' | 'Under Investigation' | 'Resolved';

type Dispute = {
  id: string;
  date: string;
  merchant: Seller;
  shipmentId: string;
  amount: number;
  reason: string;
  status: DisputeStatus;
};

const mockDisputes: Dispute[] = [
  { id: 'DIS-001', date: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(), merchant: mockSellers[0], shipmentId: 'PCL162818', amount: 500, reason: 'COD amount short', status: 'Under Investigation' },
  { id: 'DIS-002', date: new Date(Date.now() - 5 * 24 * 60 * 60 * 1000).toISOString(), merchant: mockSellers[1], shipmentId: 'PCL162823', amount: 150, reason: 'Incorrect delivery charges applied', status: 'Resolved' },
  { id: 'DIS-003', date: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000).toISOString(), merchant: mockSellers[0], shipmentId: 'PCL162820', amount: 1200, reason: 'Shipment returned but charged delivery fee', status: 'Open' },
];

export default function DisputeManagementPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const [dateRange, setDateRange] = useState<DateRange | undefined>({
    from: addDays(new Date(), -30),
    to: new Date(),
  });
  
  const filteredDisputes = useMemo(() => {
    return mockDisputes.filter(d => {
        const disputeDate = new Date(d.date);
        return dateRange?.from && dateRange?.to && disputeDate >= dateRange.from && disputeDate <= dateRange.to;
    });
  }, [dateRange]);

  const summary = useMemo(() => {
    return filteredDisputes.reduce((acc, d) => {
        if (d.status === 'Open' || d.status === 'Under Investigation') {
            acc.openCount++;
            acc.openAmount += d.amount;
        }
        if (d.status === 'Resolved') {
            acc.resolvedCount++;
        }
        return acc;
    }, { openCount: 0, openAmount: 0, resolvedCount: 0 });
  }, [filteredDisputes]);

  const getStatusBadge = (status: DisputeStatus) => {
    switch(status) {
        case 'Open': return 'bg-yellow-100 text-yellow-800';
        case 'Under Investigation': return 'bg-blue-100 text-blue-800';
        case 'Resolved': return 'bg-green-100 text-green-800';
    }
  }

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Open Disputes</CardTitle>
                        <ShieldQuestion className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{summary.openCount}</div>
                        <p className="text-xs text-muted-foreground">PKR {summary.openAmount.toLocaleString()} under review</p>
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Resolved This Month</CardTitle>
                        <CheckCircle className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">{summary.resolvedCount}</div>
                        <p className="text-xs text-muted-foreground">Disputes closed in selected period</p>
                    </CardContent>
                </Card>
            </div>
          <Card>
            <CardHeader>
              <CardTitle>Dispute Management</CardTitle>
              <CardDescription>
                Track, investigate, and resolve merchant disputes.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-8">
               <div className="p-4 border rounded-lg bg-muted/50">
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 items-end">
                      <div>
                          <label className="text-sm font-medium">Date Range</label>
                          <DatePickerWithRange date={dateRange} setDate={setDateRange} />
                      </div>
                      <div>
                          <label className="text-sm font-medium">Merchant</label>
                          <Select><SelectTrigger><SelectValue placeholder="All Merchants" /></SelectTrigger></Select>
                      </div>
                       <div>
                          <label className="text-sm font-medium">Status</label>
                          <Select><SelectTrigger><SelectValue placeholder="All Statuses" /></SelectTrigger></Select>
                      </div>
                      <Button><Search className="mr-2 h-4 w-4" /> Search</Button>
                  </div>
              </div>
              
              <div>
                <h3 className="text-lg font-semibold mb-4">Dispute Log</h3>
                 <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Date</TableHead>
                            <TableHead>Merchant</TableHead>
                            <TableHead>Shipment ID</TableHead>
                            <TableHead>Reason</TableHead>
                            <TableHead>Amount</TableHead>
                            <TableHead>Status</TableHead>
                            <TableHead className="text-right">Action</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {filteredDisputes.map(item => (
                            <TableRow key={item.id}>
                                <TableCell>{format(new Date(item.date), 'dd MMM, yyyy')}</TableCell>
                                <TableCell>{item.merchant.name}</TableCell>
                                <TableCell className="font-medium">{item.shipmentId}</TableCell>
                                <TableCell>{item.reason}</TableCell>
                                <TableCell className="font-medium">PKR {item.amount.toLocaleString()}</TableCell>
                                <TableCell>
                                    <Badge variant="outline" className={getStatusBadge(item.status)}>
                                        {item.status}
                                    </Badge>
                                </TableCell>
                                <TableCell className="text-right">
                                     <DropdownMenu>
                                        <DropdownMenuTrigger asChild>
                                            <Button size="icon" variant="ghost">
                                                <MoreVertical className="h-4 w-4" />
                                            </Button>
                                        </DropdownMenuTrigger>
                                        <DropdownMenuContent>
                                            <DropdownMenuItem>
                                                <FileText className="mr-2 h-4 w-4" /> View Details & Timeline
                                            </DropdownMenuItem>
                                            <DropdownMenuItem disabled={item.status !== 'Open'}>
                                                <Hourglass className="mr-2 h-4 w-4" /> Start Investigation
                                            </DropdownMenuItem>
                                             <DropdownMenuItem>
                                                <MessageSquarePlus className="mr-2 h-4 w-4" /> Add Note
                                            </DropdownMenuItem>
                                            <DropdownMenuItem disabled={item.status === 'Resolved'}>
                                                <CheckCircle className="mr-2 h-4 w-4" /> Mark as Resolved
                                            </DropdownMenuItem>
                                        </DropdownMenuContent>
                                    </DropdownMenu>
                                </TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
              </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}

