
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { DatePickerWithRange } from '@/components/ui/date-range-picker';
import { mockSellers, mockParcels } from '@/lib/data';
import { addDays, format } from 'date-fns';
import type { DateRange } from 'react-day-picker';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow, TableFooter as ShadTableFooter } from '@/components/ui/table';
import { Loader2, Download } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

type Transaction = {
    date: string;
    description: string;
    debit: number;
    credit: number;
};

type Statement = {
    merchantId: string;
    merchantName: string;
    dateRange: DateRange;
    openingBalance: number;
    closingBalance: number;
    transactions: Transaction[];
    summary: {
        totalCod: number;
        totalCharges: number;
        totalPayouts: number;
    };
};

export default function FinanceStatementsPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const [selectedMerchant, setSelectedMerchant] = useState<string>('');
  const [dateRange, setDateRange] = useState<DateRange | undefined>({
    from: addDays(new Date(), -30),
    to: new Date(),
  });
  const [isLoading, setIsLoading] = useState(false);
  const [statement, setStatement] = useState<Statement | null>(null);
  const { toast } = useToast();

  const handleGenerate = () => {
    if (!selectedMerchant || !dateRange?.from || !dateRange?.to) {
        toast({ variant: 'destructive', title: 'Missing Information', description: 'Please select a merchant and a date range.' });
        return;
    }
    
    setIsLoading(true);
    setTimeout(() => {
        const openingBalance = Math.random() * 5000;
        const totalCod = Math.random() * 50000;
        const totalCharges = Math.random() * -5000;
        const totalPayouts = Math.random() * -45000;
        const closingBalance = openingBalance + totalCod + totalCharges + totalPayouts;

        const transactions: Transaction[] = [
            { date: format(addDays(dateRange.from!, 5), 'yyyy-MM-dd'), description: 'Payout Batch #123', debit: 20000, credit: 0 },
            { date: format(addDays(dateRange.from!, 10), 'yyyy-MM-dd'), description: 'COD Collection (3 orders)', debit: 0, credit: 35000 },
            { date: format(addDays(dateRange.from!, 15), 'yyyy-MM-dd'), description: 'Monthly Service Charges', debit: 5000, credit: 0 },
            { date: format(addDays(dateRange.from!, 20), 'yyyy-MM-dd'), description: 'COD Collection (5 orders)', debit: 0, credit: 15000 },
            { date: format(addDays(dateRange.from!, 25), 'yyyy-MM-dd'), description: 'Payout Batch #124', debit: 25000, credit: 0 },
        ];
        

        setStatement({
            merchantId: selectedMerchant,
            merchantName: mockSellers.find(s => s.id === selectedMerchant)?.name || '',
            dateRange: dateRange,
            openingBalance,
            closingBalance,
            transactions,
            summary: { totalCod, totalCharges, totalPayouts },
        });

        setIsLoading(false);
    }, 1000);
  };

  const MetricCard = ({ title, value, className }: { title: string, value: number, className?: string }) => (
    <Card className={className}>
        <CardHeader className="p-4">
            <CardDescription>{title}</CardDescription>
            <CardTitle className="text-2xl">PKR {value.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 })}</CardTitle>
        </CardHeader>
    </Card>
  );

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Merchant Statements</CardTitle>
              <CardDescription>
                Generate and view financial statements for merchants.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <div className="p-4 border rounded-lg bg-muted/50 space-y-4">
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 items-end">
                        <div>
                            <label className="text-sm font-medium">Select Merchant</label>
                            <Select value={selectedMerchant} onValueChange={setSelectedMerchant}>
                                <SelectTrigger>
                                    <SelectValue placeholder="Select a merchant..." />
                                </SelectTrigger>
                                <SelectContent>
                                    {mockSellers.map(seller => (
                                        <SelectItem key={seller.id} value={seller.id}>{seller.name}</SelectItem>
                                    ))}
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="md:col-span-1">
                            <label className="text-sm font-medium">Select Date Range</label>
                            <DatePickerWithRange date={dateRange} setDate={setDateRange} />
                        </div>
                        <Button onClick={handleGenerate} disabled={isLoading} className="w-full md:w-auto">
                            {isLoading && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                            {isLoading ? 'Generating...' : 'Generate Statement'}
                        </Button>
                    </div>
                </div>
            </CardContent>
          </Card>
           {statement && (
                <Card className="animate-in fade-in-50">
                    <CardHeader>
                        <div className="flex justify-between items-start">
                            <div>
                                <h3 className="text-xl font-semibold">Statement for {statement.merchantName}</h3>
                                <p className="text-sm text-muted-foreground">
                                    Period: {statement.dateRange?.from?.toLocaleDateString()} - {statement.dateRange?.to?.toLocaleDateString()}
                                </p>
                            </div>
                            <Button variant="outline"><Download className="mr-2 h-4 w-4" /> Download PDF</Button>
                        </div>
                    </CardHeader>
                    <CardContent>
                         <div className="grid grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                            <MetricCard title="Opening Balance" value={statement.openingBalance} />
                            <MetricCard title="Total Credits (COD)" value={statement.summary.totalCod} className="border-green-500" />
                            <MetricCard title="Total Debits (Payouts/Charges)" value={statement.summary.totalPayouts + statement.summary.totalCharges} className="border-red-500" />
                            <MetricCard title="Closing Balance" value={statement.closingBalance} />
                        </div>

                         <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>Date</TableHead>
                                    <TableHead>Description</TableHead>
                                    <TableHead className="text-right">Debit</TableHead>
                                    <TableHead className="text-right">Credit</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {statement.transactions.map((tx, index) => (
                                    <TableRow key={index}>
                                        <TableCell>{tx.date}</TableCell>
                                        <TableCell>{tx.description}</TableCell>
                                        <TableCell className="text-right font-mono text-red-600">{tx.debit > 0 ? tx.debit.toFixed(2) : '-'}</TableCell>
                                        <TableCell className="text-right font-mono text-green-600">{tx.credit > 0 ? tx.credit.toFixed(2) : '-'}</TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                            <ShadTableFooter>
                                <TableRow>
                                    <TableCell colSpan={2} className="font-bold text-right">Closing Balance</TableCell>
                                    <TableCell colSpan={2} className="text-right font-bold">
                                        PKR {statement.closingBalance.toLocaleString(undefined, { minimumFractionDigits: 2, maximumFractionDigits: 2 })}
                                    </TableCell>
                                </TableRow>
                            </ShadTableFooter>
                        </Table>
                    </CardContent>
                </Card>
            )}
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
