
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Calendar } from '@/components/ui/calendar';
import { Badge } from '@/components/ui/badge';
import { format } from 'date-fns';

type TaxEvent = {
  date: Date;
  title: string;
  type: 'GST' | 'Income Tax' | 'Corporate';
};

const mockEvents: TaxEvent[] = [
  { date: new Date('2024-08-15'), title: 'GST Filing for July', type: 'GST' },
  { date: new Date('2024-09-15'), title: 'GST Filing for August', type: 'GST' },
  { date: new Date('2024-09-30'), title: 'Annual Income Tax Return', type: 'Income Tax' },
  { date: new Date('2024-10-15'), title: 'GST Filing for September', type: 'GST' },
];

const typeColors = {
    GST: 'bg-blue-100 text-blue-800',
    'Income Tax': 'bg-green-100 text-green-800',
    Corporate: 'bg-purple-100 text-purple-800',
}

export default function TaxCalendarPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [date, setDate] = useState<Date | undefined>(new Date());
  
  const eventsForMonth = mockEvents.filter(
    event => date && event.date.getMonth() === date.getMonth() && event.date.getFullYear() === date.getFullYear()
  ).sort((a,b) => a.date.getDate() - b.date.getDate());

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Tax Compliance Calendar</CardTitle>
              <CardDescription>
                An overview of all important tax deadlines and filing dates.
              </CardDescription>
            </CardHeader>
            <CardContent className="grid md:grid-cols-[1fr_auto] gap-8">
              <div>
                <h3 className="text-lg font-semibold mb-4">
                  Deadlines for {date ? format(date, 'MMMM yyyy') : ''}
                </h3>
                <div className="space-y-4">
                    {eventsForMonth.length > 0 ? eventsForMonth.map(event => (
                        <div key={event.title} className="flex items-start gap-4 p-3 rounded-lg bg-muted/50">
                            <div className="flex flex-col items-center justify-center w-12 h-12 rounded-md bg-background border">
                                <span className="text-sm font-semibold text-primary">{format(event.date, 'MMM')}</span>
                                <span className="text-xl font-bold">{format(event.date, 'dd')}</span>
                            </div>
                            <div>
                                <p className="font-semibold">{event.title}</p>
                                <Badge variant="outline" className={`mt-1 ${typeColors[event.type]}`}>{event.type}</Badge>
                            </div>
                        </div>
                    )) : (
                        <p className="text-muted-foreground">No deadlines for this month.</p>
                    )}
                </div>
              </div>
              <div className="flex justify-center">
                <Calendar
                    mode="single"
                    selected={date}
                    onSelect={setDate}
                    className="rounded-md border"
                    components={{
                        DayContent: ({ date }) => {
                           const event = mockEvents.find(e => e.date.toDateString() === date.toDateString());
                           return (
                                <>
                                 {format(date, 'd')}
                                 {event && <div className="absolute bottom-1 left-1/2 -translate-x-1/2 h-1 w-1 rounded-full bg-primary" />}
                                </>
                           );
                        }
                    }}
                />
              </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
