
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Download, Percent, FileText } from 'lucide-react';
import { DatePickerWithRange } from '@/components/ui/date-range-picker';
import type { DateRange } from 'react-day-picker';
import { addDays, format } from 'date-fns';

type GstTransaction = {
  id: string;
  date: string;
  description: string;
  amount: number;
  gstAmount: number;
  type: 'Input' | 'Output';
};

const mockTransactions: GstTransaction[] = [
  { id: 'gst-1', date: new Date(Date.now() - 2 * 24 * 60 * 60 * 1000).toISOString(), description: 'Delivery Charges - Batch 001', amount: 50000, gstAmount: 8000, type: 'Output' },
  { id: 'gst-2', date: new Date(Date.now() - 3 * 24 * 60 * 60 * 1000).toISOString(), description: 'Fuel Purchase - Invoice #FP-123', amount: 25000, gstAmount: 4000, type: 'Input' },
  { id: 'gst-3', date: new Date(Date.now() - 5 * 24 * 60 * 60 * 1000).toISOString(), description: 'Packaging Material - Inv #PKG-456', amount: 15000, gstAmount: 2400, type: 'Input' },
  { id: 'gst-4', date: new Date(Date.now() - 6 * 24 * 60 * 60 * 1000).toISOString(), description: 'Delivery Charges - Batch 002', amount: 75000, gstAmount: 12000, type: 'Output' },
];

export default function GstVatPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const [dateRange, setDateRange] = useState<DateRange | undefined>({
    from: addDays(new Date(), -30),
    to: new Date(),
  });

  const { totalInput, totalOutput, netPayable } = useMemo(() => {
    const output = mockTransactions
      .filter(tx => tx.type === 'Output')
      .reduce((sum, tx) => sum + tx.gstAmount, 0);
    const input = mockTransactions
      .filter(tx => tx.type === 'Input')
      .reduce((sum, tx) => sum + tx.gstAmount, 0);
    return {
      totalInput: input,
      totalOutput: output,
      netPayable: output - input,
    };
  }, []);


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
            <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Output GST (Collected)</CardTitle>
                        <Percent className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">PKR {totalOutput.toLocaleString()}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Input GST (Paid)</CardTitle>
                        <Percent className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">PKR {totalInput.toLocaleString()}</div>
                    </CardContent>
                </Card>
                <Card>
                    <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                        <CardTitle className="text-sm font-medium">Net GST Payable</CardTitle>
                        <FileText className="h-4 w-4 text-muted-foreground" />
                    </CardHeader>
                    <CardContent>
                        <div className="text-2xl font-bold">PKR {netPayable.toLocaleString()}</div>
                    </CardContent>
                </Card>
            </div>
          <Card>
            <CardHeader className="flex flex-col md:flex-row md:items-center justify-between gap-4">
              <div className="grid gap-2">
                <CardTitle>GST Calculation & Reporting</CardTitle>
                <CardDescription>
                  Review GST collected on sales and paid on purchases to determine net liability.
                </CardDescription>
              </div>
              <div className="flex items-center gap-2">
                <DatePickerWithRange date={dateRange} setDate={setDateRange} />
                <Button variant="outline"><Download className="mr-2 h-4 w-4" /> Export Report</Button>
              </div>
            </CardHeader>
            <CardContent>
               <Table>
                    <TableHeader>
                        <TableRow>
                            <TableHead>Date</TableHead>
                            <TableHead>Description</TableHead>
                            <TableHead>Type</TableHead>
                            <TableHead>Transaction Amount</TableHead>
                            <TableHead className="text-right">GST Amount</TableHead>
                        </TableRow>
                    </TableHeader>
                    <TableBody>
                        {mockTransactions.map(tx => (
                            <TableRow key={tx.id}>
                                <TableCell>{format(new Date(tx.date), 'dd MMM, yyyy')}</TableCell>
                                <TableCell>{tx.description}</TableCell>
                                <TableCell>
                                    <Badge variant={tx.type === 'Input' ? 'secondary' : 'default'}>{tx.type}</Badge>
                                </TableCell>
                                <TableCell>PKR {tx.amount.toLocaleString()}</TableCell>
                                <TableCell className="text-right font-medium">PKR {tx.gstAmount.toLocaleString()}</TableCell>
                            </TableRow>
                        ))}
                    </TableBody>
                </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
