
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Download, Users, Briefcase } from 'lucide-react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { format } from 'date-fns';

type WithholdingEntry = {
  id: string;
  entityType: 'Employee' | 'Vendor';
  entityName: string;
  paymentDate: string;
  grossAmount: number;
  withheldAmount: number;
};

const mockWithholding: WithholdingEntry[] = [
  { id: 'wh-1', entityType: 'Employee', entityName: 'Ahmed Hassan', paymentDate: '2024-07-31', grossAmount: 50000, withheldAmount: 1500 },
  { id: 'wh-2', entityType: 'Vendor', entityName: 'Punjab Packaging', paymentDate: '2024-07-25', grossAmount: 25000, withheldAmount: 500 },
  { id: 'wh-3', entityType: 'Employee', entityName: 'Aisha Bibi', paymentDate: '2024-07-31', grossAmount: 48000, withheldAmount: 1200 },
];

export default function WithholdingTaxPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader className="flex flex-col md:flex-row md:items-center justify-between gap-4">
              <div className="grid gap-2">
                <CardTitle>Income Tax Withholding</CardTitle>
                <CardDescription>
                  Track and report income tax withheld from employee salaries and vendor payments.
                </CardDescription>
              </div>
               <div className="flex items-center gap-2">
                 <Select defaultValue="all">
                    <SelectTrigger className="w-[180px]"><SelectValue /></SelectTrigger>
                    <SelectContent>
                        <SelectItem value="all">All Entity Types</SelectItem>
                        <SelectItem value="employee">Employees</SelectItem>
                        <SelectItem value="vendor">Vendors</SelectItem>
                    </SelectContent>
                 </Select>
                 <Button variant="outline"><Download className="mr-2 h-4 w-4" /> Export</Button>
               </div>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                    <TableRow>
                        <TableHead>Entity Name</TableHead>
                        <TableHead>Type</TableHead>
                        <TableHead>Payment Date</TableHead>
                        <TableHead>Gross Amount</TableHead>
                        <TableHead className="text-right">Tax Withheld</TableHead>
                    </TableRow>
                </TableHeader>
                <TableBody>
                    {mockWithholding.map(entry => (
                        <TableRow key={entry.id}>
                            <TableCell className="font-medium">{entry.entityName}</TableCell>
                            <TableCell>
                                {entry.entityType === 'Employee' ? 
                                    <Users className="h-4 w-4 text-muted-foreground" /> : 
                                    <Briefcase className="h-4 w-4 text-muted-foreground" />}
                            </TableCell>
                            <TableCell>{format(new Date(entry.paymentDate), 'dd MMM, yyyy')}</TableCell>
                            <TableCell>PKR {entry.grossAmount.toLocaleString()}</TableCell>
                            <TableCell className="text-right font-medium">PKR {entry.withheldAmount.toLocaleString()}</TableCell>
                        </TableRow>
                    ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
