
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { DatePicker } from '@/components/ui/date-picker';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Textarea } from '@/components/ui/textarea';
import { Save, Printer, RefreshCw, ArrowRight } from 'lucide-react';
import { useToast } from '@/hooks/use-toast';

const cashAndBankAccounts = [
    { id: 'cash', name: 'Cash Account' },
    { id: 'hbl-1', name: 'HBL - ****1234' },
    { id: 'mcb-1', name: 'MCB - ****5678' },
];

export default function ContraVoucherPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [date, setDate] = useState<Date | undefined>(new Date());
  const [amount, setAmount] = useState('');
  const [fromAccount, setFromAccount] = useState('');
  const [toAccount, setToAccount] = useState('');
  const [remarks, setRemarks] = useState('');
  const { toast } = useToast();

  const handleSave = () => {
    if (!fromAccount || !toAccount || !amount || parseFloat(amount) <= 0) {
        toast({ variant: 'destructive', title: 'Error', description: 'Please select accounts and enter a valid amount.' });
        return;
    }
    if (fromAccount === toAccount) {
        toast({ variant: 'destructive', title: 'Error', description: 'Source and destination accounts cannot be the same.' });
        return;
    }
    toast({ title: 'Success', description: 'Contra Voucher saved successfully!' });
  };
  
  const handleReset = () => {
    setDate(new Date());
    setAmount('');
    setFromAccount('');
    setToAccount('');
    setRemarks('');
  }

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="finance">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Contra Voucher</CardTitle>
              <CardDescription>
                Record transfers between cash and bank accounts.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                        <Label>Voucher No.</Label>
                        <Input readOnly value="CV-AUTO-00034" className="bg-muted/50" />
                    </div>
                     <div>
                        <Label>Date</Label>
                        <DatePicker date={date} setDate={setDate} />
                    </div>
                </div>

                <div className="space-y-4 max-w-4xl mx-auto p-6 border rounded-lg">
                    <div className="grid grid-cols-1 md:grid-cols-[1fr_auto_1fr] items-end gap-4">
                         <div>
                            <Label htmlFor="from-account">From Account (Credit)</Label>
                            <Select value={fromAccount} onValueChange={setFromAccount}>
                                <SelectTrigger id="from-account"><SelectValue placeholder="Select source..." /></SelectTrigger>
                                <SelectContent>
                                    {cashAndBankAccounts.map(acc => <SelectItem key={acc.id} value={acc.id}>{acc.name}</SelectItem>)}
                                </SelectContent>
                            </Select>
                        </div>
                        <div className="self-center">
                            <ArrowRight className="h-6 w-6 text-muted-foreground" />
                        </div>
                         <div>
                            <Label htmlFor="to-account">To Account (Debit)</Label>
                            <Select value={toAccount} onValueChange={setToAccount}>
                                <SelectTrigger id="to-account"><SelectValue placeholder="Select destination..." /></SelectTrigger>
                                <SelectContent>
                                    {cashAndBankAccounts.map(acc => <SelectItem key={acc.id} value={acc.id}>{acc.name}</SelectItem>)}
                                </SelectContent>
                            </Select>
                        </div>
                    </div>
                    <div>
                        <Label htmlFor="amount">Amount (PKR)</Label>
                        <Input id="amount" type="number" placeholder="Enter amount to transfer" value={amount} onChange={e => setAmount(e.target.value)} />
                    </div>
                </div>

                <div>
                    <Label htmlFor="remarks">Remarks / Narration</Label>
                    <Textarea id="remarks" placeholder="e.g., Cash deposited to HBL account" value={remarks} onChange={e => setRemarks(e.target.value)} />
                </div>
            </CardContent>
            <CardFooter className="justify-end gap-2 border-t pt-6">
                <Button variant="outline" onClick={handleReset}><RefreshCw className="mr-2 h-4 w-4" /> Reset</Button>
                <Button variant="secondary"><Printer className="mr-2 h-4 w-4" /> Save & Print</Button>
                <Button onClick={handleSave}><Save className="mr-2 h-4 w-4" /> Save Voucher</Button>
            </CardFooter>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
