
'use client';

import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { MessageSquareWarning, FileClock, RefreshCcw } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { formatDistanceToNow } from 'date-fns';

type AlertType = 'Complaint' | 'Contract' | 'RTO';

type Alert = {
  id: string;
  type: AlertType;
  title: string;
  description: string;
  timestamp: Date;
  priority: 'High' | 'Medium' | 'Low';
};

const alerts: Alert[] = [
  {
    id: 'alert-1',
    type: 'Complaint',
    title: 'SLA Breach: Ticket #CMP-002',
    description: 'Complaint from Gadget World regarding a delayed shipment has breached the 24-hour resolution SLA.',
    timestamp: new Date(Date.now() - 2 * 60 * 60 * 1000),
    priority: 'High',
  },
  {
    id: 'alert-5',
    type: 'RTO',
    title: 'High RTO Rate: Chic Boutique',
    description: 'Chic Boutique has a return-to-origin (RTO) rate of 25% this month, which is above the 15% threshold.',
    timestamp: new Date(Date.now() - 22 * 60 * 60 * 1000),
    priority: 'Medium',
  },
  {
    id: 'alert-3',
    type: 'Contract',
    title: 'Contract Expiring Soon: Chic Boutique',
    description: 'The annual contract for Chic Boutique is set to expire in 15 days. Action is required for renewal.',
    timestamp: new Date(Date.now() - 1 * 24 * 60 * 60 * 1000),
    priority: 'Medium',
  },
   {
    id: 'alert-4',
    type: 'Complaint',
    title: 'New High-Priority Ticket: #CMP-005',
    description: 'A new complaint has been lodged by "Fashion Forward" regarding a damaged parcel.',
    timestamp: new Date(Date.now() - 30 * 60 * 1000),
    priority: 'High',
  },
];

const priorityStyles = {
  High: 'border-red-500 bg-red-50 dark:bg-red-900/20',
  Medium: 'border-yellow-500 bg-yellow-50 dark:bg-yellow-900/20',
  Low: 'border-gray-300',
};

const iconMap: Record<AlertType, React.ElementType> = {
    Complaint: MessageSquareWarning,
    Contract: FileClock,
    RTO: RefreshCcw,
}

export default function AdminMarketingAlertsPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="marketing-manager">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Critical Alerts</CardTitle>
              <CardDescription>
                A real-time feed of critical alerts regarding complaints, contracts, and RTO rates that require immediate attention.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <div className="grid gap-6">
                    {alerts.map((alert) => {
                        const Icon = iconMap[alert.type];
                        return (
                             <Card key={alert.id} className={priorityStyles[alert.priority]}>
                                <CardHeader className="flex flex-row items-start gap-4 space-y-0 pb-4">
                                    <div className="flex h-10 w-10 items-center justify-center rounded-full bg-muted">
                                        <Icon className="h-5 w-5 text-muted-foreground" />
                                    </div>
                                    <div className="flex-1">
                                        <div className="flex items-center justify-between">
                                            <CardTitle className="text-base">{alert.title}</CardTitle>
                                            <Badge variant={alert.priority === 'High' ? 'destructive' : 'secondary'}>
                                                {alert.priority} Priority
                                            </Badge>
                                        </div>
                                        <CardDescription className="mt-1">{alert.description}</CardDescription>
                                    </div>
                                </CardHeader>
                                <CardFooter className="text-xs text-muted-foreground">
                                    {formatDistanceToNow(alert.timestamp, { addSuffix: true })}
                                </CardFooter>
                            </Card>
                        )
                    })}
                </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
