
'use client';

import { useState, useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Seller } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { MoreVertical, MessageSquarePlus, CheckCircle, Hourglass } from 'lucide-react';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger } from '@/components/ui/dropdown-menu';
import { mockSellers } from '@/lib/data';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';


type DisputeType = 'Short Payment' | 'COD Payout Delay' | 'Wrong Invoice' | 'COD Pending > 7 Days';
type DisputeStatus = 'Open' | 'Under Investigation' | 'Resolved';

type Dispute = {
  id: string;
  merchant: Seller;
  type: DisputeType;
  amount: number;
  date: string;
  status: DisputeStatus;
  relatedShipment?: string;
};

const mockDisputes: Dispute[] = [
  { id: 'DIS-FIN-001', merchant: mockSellers[0], type: 'Short Payment', amount: 500, date: '2024-07-28', status: 'Under Investigation', relatedShipment: 'PCL162818' },
  { id: 'DIS-FIN-002', merchant: mockSellers[1], type: 'COD Payout Delay', amount: 12500, date: '2024-07-27', status: 'Open', },
  { id: 'DIS-FIN-003', merchant: mockSellers[0], type: 'Wrong Invoice', amount: 150, date: '2024-07-26', status: 'Resolved' },
  { id: 'DIS-FIN-004', merchant: mockSellers[1], type: 'COD Pending > 7 Days', amount: 8300, date: '2024-07-25', status: 'Open' },
];

export default function AdminMarketingFinanceDisputesPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const [activeTab, setActiveTab] = useState<DisputeType | 'All'>('All');

  const filteredDisputes = useMemo(() => {
    if (activeTab === 'All') return mockDisputes;
    return mockDisputes.filter(d => d.type === activeTab);
  }, [activeTab]);

  const getStatusBadge = (status: DisputeStatus) => {
    switch (status) {
      case 'Open': return 'bg-yellow-100 text-yellow-800';
      case 'Under Investigation': return 'bg-blue-100 text-blue-800';
      case 'Resolved': return 'bg-green-100 text-green-800';
      default: return 'bg-gray-100 text-gray-800';
    }
  }

  const DisputeTable = ({ disputes }: { disputes: Dispute[] }) => (
      <Table>
        <TableHeader>
          <TableRow>
            <TableHead>Dispute ID</TableHead>
            <TableHead>Merchant</TableHead>
            <TableHead>Type/Reason</TableHead>
            <TableHead>Amount</TableHead>
            <TableHead>Status</TableHead>
            <TableHead className="text-right">Actions</TableHead>
          </TableRow>
        </TableHeader>
        <TableBody>
          {disputes.map(dispute => (
            <TableRow key={dispute.id}>
              <TableCell className="font-medium">{dispute.id}</TableCell>
              <TableCell>{dispute.merchant.name}</TableCell>
              <TableCell>
                <div>{dispute.type}</div>
                {dispute.relatedShipment && (
                  <div className="text-xs text-muted-foreground">Shipment: {dispute.relatedShipment}</div>
                )}
              </TableCell>
               <TableCell>PKR {dispute.amount.toLocaleString()}</TableCell>
              <TableCell>
                <Badge variant="outline" className={getStatusBadge(dispute.status)}>{dispute.status}</Badge>
              </TableCell>
              <TableCell className="text-right">
                 <DropdownMenu>
                    <DropdownMenuTrigger asChild>
                        <Button size="icon" variant="ghost">
                            <MoreVertical className="h-4 w-4" />
                        </Button>
                    </DropdownMenuTrigger>
                    <DropdownMenuContent>
                        <DropdownMenuItem>
                            <Hourglass className="mr-2 h-4 w-4" /> Start Investigation
                        </DropdownMenuItem>
                         <DropdownMenuItem>
                            <MessageSquarePlus className="mr-2 h-4 w-4" /> Add Note
                        </DropdownMenuItem>
                        <DropdownMenuItem>
                            <CheckCircle className="mr-2 h-4 w-4" /> Mark as Resolved
                        </DropdownMenuItem>
                    </DropdownMenuContent>
                </DropdownMenu>
              </TableCell>
            </TableRow>
          ))}
        </TableBody>
      </Table>
  );

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="marketing-manager">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader>
              <CardTitle>Finance Dispute Resolver</CardTitle>
              <CardDescription>
                Manage disputes related to short payments, COD delays, and incorrect invoices.
              </CardDescription>
            </CardHeader>
            <CardContent>
                <Tabs defaultValue="All" onValueChange={(value) => setActiveTab(value as any)}>
                    <TabsList className="grid w-full grid-cols-5">
                        <TabsTrigger value="All">All</TabsTrigger>
                        <TabsTrigger value="Short Payment">Short Payment</TabsTrigger>
                        <TabsTrigger value="COD Payout Delay">COD Payout Delay</TabsTrigger>
                        <TabsTrigger value="Wrong Invoice">Wrong Invoice</TabsTrigger>
                        <TabsTrigger value="COD Pending > 7 Days">Automated Alerts</TabsTrigger>
                    </TabsList>
                    <TabsContent value="All" className="mt-6">
                        <DisputeTable disputes={filteredDisputes} />
                    </TabsContent>
                    <TabsContent value="Short Payment" className="mt-6">
                        <DisputeTable disputes={filteredDisputes} />
                    </TabsContent>
                    <TabsContent value="COD Payout Delay" className="mt-6">
                       <DisputeTable disputes={filteredDisputes} />
                    </TabsContent>
                    <TabsContent value="Wrong Invoice" className="mt-6">
                        <DisputeTable disputes={filteredDisputes} />
                    </TabsContent>
                    <TabsContent value="COD Pending > 7 Days" className="mt-6">
                        <DisputeTable disputes={filteredDisputes} />
                    </TabsContent>
                </Tabs>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
