
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Users, PlusCircle } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Avatar, AvatarFallback, AvatarImage } from '@/components/ui/avatar';

type LeadStatus = 'Prospect' | 'Demo' | 'Onboard' | 'Active';

type Lead = {
  id: string;
  companyName: string;
  contactName: string;
  value: number;
  status: LeadStatus;
  avatar: string;
};

const initialLeads: Lead[] = [
  { id: 'lead-1', companyName: 'Fashion Hub', contactName: 'Ali Raza', value: 50000, status: 'Prospect', avatar: 'https://picsum.photos/seed/lead1/100' },
  { id: 'lead-2', companyName: 'Electronics Direct', contactName: 'Sara Khan', value: 75000, status: 'Prospect', avatar: 'https://picsum.photos/seed/lead2/100' },
  { id: 'lead-3', companyName: 'Home Goods Inc.', contactName: 'Fatima Ahmed', value: 120000, status: 'Demo', avatar: 'https://picsum.photos/seed/lead3/100' },
  { id: 'lead-4', companyName: 'Beauty Essentials', contactName: 'Bilal Hussain', value: 30000, status: 'Demo', avatar: 'https://picsum.photos/seed/lead4/100' },
  { id: 'lead-5', companyName: 'Sports Gear Pro', contactName: 'Ayesha Malik', value: 90000, status: 'Onboard', avatar: 'https://picsum.photos/seed/lead5/100' },
  { id: 'lead-6', companyName: 'Book Worms', contactName: 'Zainab Omar', value: 25000, status: 'Active', avatar: 'https://picsum.photos/seed/lead6/100' },
];

const pipelineStages: LeadStatus[] = ['Prospect', 'Demo', 'Onboard', 'Active'];

export default function AdminMarketingLeadManagementPage() {
  
  const handleBookingComplete = (result: BookingResult) => {};
  const [leads, setLeads] = useState(initialLeads);

  const LeadCard = ({ lead }: { lead: Lead }) => (
    <Card className="mb-4 cursor-grab active:cursor-grabbing">
      <CardContent className="p-4">
        <div className="flex items-start gap-4">
          <Avatar className="h-10 w-10 border">
            <AvatarImage src={lead.avatar} />
            <AvatarFallback>{lead.companyName.charAt(0)}</AvatarFallback>
          </Avatar>
          <div className="flex-1">
            <p className="font-semibold">{lead.companyName}</p>
            <p className="text-sm text-muted-foreground">{lead.contactName}</p>
            <p className="text-sm font-medium text-primary mt-1">PKR {lead.value.toLocaleString()}</p>
          </div>
        </div>
      </CardContent>
    </Card>
  );

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="marketing-manager">
        <div className="flex flex-col gap-8">
          <Card>
            <CardHeader className="flex flex-row items-center">
              <div className="grid gap-2">
                <CardTitle>Lead Management</CardTitle>
                <CardDescription>
                  Track leads from prospect to active merchant.
                </CardDescription>
              </div>
              <Button size="sm" className="ml-auto gap-1">
                <PlusCircle className="h-4 w-4" />
                Add Lead
              </Button>
            </CardHeader>
            <CardContent>
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                    {pipelineStages.map(stage => (
                        <div key={stage} className="bg-muted/50 rounded-lg p-4">
                            <h3 className="font-semibold mb-4 text-center">{stage}</h3>
                            <div className="space-y-4">
                                {leads.filter(lead => lead.status === stage).map(lead => (
                                    <LeadCard key={lead.id} lead={lead} />
                                ))}
                                {leads.filter(lead => lead.status === stage).length === 0 && (
                                    <div className="text-center text-sm text-muted-foreground py-10">
                                        No leads in this stage.
                                    </div>
                                )}
                            </div>
                        </div>
                    ))}
                </div>
            </CardContent>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
