
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardHeader, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { DatePicker } from '@/components/ui/date-picker';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Search, RefreshCw, Printer, Download, Settings, MoreVertical, UploadCloud } from 'lucide-react';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger, DropdownMenuSub, DropdownMenuSubTrigger, DropdownMenuSubContent, DropdownMenuRadioGroup, DropdownMenuRadioItem, DropdownMenuLabel } from '@/components/ui/dropdown-menu';

type CprEntry = {
  id: string;
  cprNumber: string;
  date: string;
  amount: number;
  status: 'New' | 'Closed';
};

const mockCprs: CprEntry[] = [];


export default function CashPaymentReceiptPage() {
  const [cprs, setCprs] = useState<CprEntry[]>(mockCprs);
  const handleBookingComplete = (result: BookingResult) => {};

  const getStatusVariant = (status: CprEntry['status']) => {
    switch(status) {
        case 'New': return 'bg-green-100 text-green-800';
        case 'Closed': return 'bg-gray-200 text-gray-800';
        default: return 'bg-secondary';
    }
  }

  const [showCount, setShowCount] = useState("25");
  const [sortOrder, setSortOrder] = useState("desc");

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="seller">
        <div className="flex flex-col gap-8">
            <Card>
                <CardHeader>
                    <div className="flex flex-wrap items-center justify-between gap-4">
                        <div className="flex items-center gap-2 border-b">
                            <Button variant="ghost" className="border-b-2 border-primary text-primary rounded-none">All ({cprs.length})</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none">New ({cprs.filter(c => c.status === 'New').length})</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none">Closed ({cprs.filter(c => c.status === 'Closed').length})</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none">Special Request (0)</Button>
                        </div>
                        <div className="flex items-center gap-2">
                            <Button variant="outline" size="icon"><RefreshCw className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><UploadCloud className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><Download className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><Printer className="h-4 w-4" /></Button>
                             <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <Button variant="outline" size="icon">
                                        <Settings className="h-4 w-4" />
                                    </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                    <DropdownMenuLabel>Show</DropdownMenuLabel>
                                    <DropdownMenuRadioGroup value={showCount} onValueChange={setShowCount}>
                                        <DropdownMenuRadioItem value="25">25</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="50">50</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="100">100</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="200">200</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="300">300</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="all">All</DropdownMenuRadioItem>
                                    </DropdownMenuRadioGroup>
                                    <DropdownMenuLabel className="mt-2">Sort By Order Date</DropdownMenuLabel>
                                    <DropdownMenuRadioGroup value={sortOrder} onValueChange={setSortOrder}>
                                        <DropdownMenuRadioItem value="asc">ASC</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="desc">DESC</DropdownMenuRadioItem>
                                    </DropdownMenuRadioGroup>
                                </DropdownMenuContent>
                            </DropdownMenu>
                        </div>
                    </div>
                </CardHeader>
                <CardContent className="space-y-6">
                    <div className="p-4 border rounded-lg bg-muted/50">
                        <p className="text-sm font-semibold mb-4">ADVANCED SEARCH</p>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 items-end">
                            <div className="lg:col-span-1">
                                <label className="text-xs text-muted-foreground">Search by</label>
                                <div className="flex">
                                    <Input placeholder="Search..." className="rounded-r-none" />
                                    <Button variant="secondary" className="rounded-l-none border-l-0">CPR #</Button>
                                </div>
                            </div>
                            <div>
                                <label className="text-xs text-muted-foreground">FROM</label>
                                <DatePicker />
                            </div>
                            <div>
                                <label className="text-xs text-muted-foreground">TO</label>
                                <DatePicker />
                            </div>
                            <div className="flex items-end gap-2">
                                <Button className="w-full md:w-auto"><Search className="mr-2 h-4 w-4"/> Search</Button>
                                <Button variant="secondary" className="w-full md:w-auto">Clear Filter</Button>
                            </div>
                        </div>
                    </div>

                    <div className="overflow-x-auto">
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>CPR NUMBER</TableHead>
                                    <TableHead>AMOUNT</TableHead>
                                    <TableHead>STATUS</TableHead>
                                    <TableHead>ACTION</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {cprs.length === 0 ? (
                                    <TableRow>
                                        <TableCell colSpan={4} className="h-24 text-center">No CPRs to display.</TableCell>
                                    </TableRow>
                                ) : cprs.map(cpr => (
                                    <TableRow key={cpr.id}>
                                        <TableCell>
                                            <div className="font-medium text-primary">{cpr.cprNumber}</div>
                                            <div className="text-xs text-muted-foreground">{cpr.date}</div>
                                        </TableCell>
                                        <TableCell>{cpr.amount.toFixed(2)}</TableCell>
                                        <TableCell>
                                            <Badge variant="outline" className={`${getStatusVariant(cpr.status)} capitalize`}>{cpr.status}</Badge>
                                        </TableCell>
                                        <TableCell>
                                            <DropdownMenu>
                                                <DropdownMenuTrigger asChild>
                                                    <Button variant="ghost" size="icon" className="h-8 w-8">
                                                        <MoreVertical className="h-4 w-4" />
                                                    </Button>
                                                </DropdownMenuTrigger>
                                                <DropdownMenuContent align="end">
                                                    <DropdownMenuItem>View Details</DropdownMenuItem>
                                                    <DropdownMenuItem>Download PDF</DropdownMenuItem>
                                                </DropdownMenuContent>
                                            </DropdownMenu>
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </div>
                </CardContent>
                <CardFooter className="flex-col items-start gap-4 sm:flex-row sm:items-center sm:justify-between">
                     <p className="text-sm text-muted-foreground">Showing {cprs.length} of {cprs.length} entries</p>
                    <div className="flex items-center gap-2">
                        <span>Page</span>
                        <Select defaultValue="1">
                            <SelectTrigger className="w-16"><SelectValue /></SelectTrigger>
                            <SelectContent>
                                <SelectItem value="1">1</SelectItem>
                            </SelectContent>
                        </Select>
                        <span className="text-sm">of 1</span>
                        <Button variant="outline" size="sm" disabled>Previous</Button>
                        <Button variant="outline" size="sm" disabled>Next</Button>
                    </div>
                </CardFooter>
            </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
