
'use server';

import { mockUser, historicalDeliveryData } from '@/lib/data';
import type { BookingResult, Parcel, ParcelFormValues } from '@/lib/types';
import { predictDeliveryETA, type PredictDeliveryETAInput } from '@/ai/flows/delivery-eta-prediction';

export async function bookParcelAction(values: ParcelFormValues): Promise<BookingResult> {
  // 1. "Save" the parcel (mock implementation)
  const newParcel: Parcel = {
    id: `PCL${Date.now().toString().slice(-6)}`,
    recipientName: values.customerName,
    address: values.deliveryAddress,
    contact: values.customerContact,
    codAmount: values.orderAmount,
    city: values.deliveryCity,
    zone: values.zone,
    bookingDate: new Date(values.orderDate).toISOString(),
    status: 'Pending',
    sellerId: mockUser.id,
    orderType: values.orderType,
    orderReferenceNumber: values.orderReferenceNumber,
    items: values.items,
    bookingWeight: values.bookingWeight,
    pickupCity: values.pickupCity,
    pickupAddress: values.pickupAddress,
    returnCity: values.returnCity,
    returnAddress: values.returnAddress,
    orderDetail: values.orderDetail,
    notes: values.notes,
  };

  // 2. Prepare data for ETA prediction
  const etaInput: PredictDeliveryETAInput = {
    parcelId: newParcel.id,
    city: newParcel.city,
    zone: newParcel.zone,
    bookingDate: newParcel.bookingDate,
    historicalData: JSON.stringify(historicalDeliveryData),
  };

  try {
    // 3. Call the AI flow
    const prediction = await predictDeliveryETA(etaInput);
    
    // 4. Return both parcel and prediction
    return { parcel: newParcel, eta: prediction };
  } catch (error) {
    console.error("Error predicting ETA:", error);
    
    // Handle error, return a successful booking without a valid ETA
    return {
      parcel: newParcel,
      eta: {
        estimatedDeliveryTime: "N/A",
        confidence: 0,
        reasoning: "Could not predict ETA due to an internal error. The parcel has been booked successfully."
      }
    };
  }
}
