

'use client';

import { useState, useEffect } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel, Complaint, NewsItem } from '@/lib/types';
import { mockParcels, mockUser, mockComplaints, mockNews } from '@/lib/data';
import { bookParcelAction } from '@/app/dashboard/actions';
import { OverviewCards } from '@/components/dashboard/overview-cards';
import { Button } from '@/components/ui/button';
import { PlusCircle, Search } from 'lucide-react';
import { ParcelsTable } from '@/components/dashboard/parcels-table';
import { Card, CardContent, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { DeliveryPerformanceChart } from '@/components/dashboard/delivery-performance-chart';
import Link from 'next/link';
import { NewTrackParcelDialog, TrackParcelDialog } from '@/components/track-parcel-dialog';
import { BookParcelDialog } from '@/components/book-parcel-dialog';
import { ETAPredictionDialog } from '@/components/eta-prediction-dialog';
import { ComplaintsPreview } from '@/components/dashboard/complaints-preview';
import { AnnouncementDialog } from '@/components/announcement-dialog';

export default function SellerDashboard() {
  const [parcels, setParcels] = useState<Parcel[]>([]);
  const [complaints, setComplaints] = useState<Complaint[]>([]);
  const [bookingResult, setBookingResult] = useState<BookingResult>(null);
  const [trackedParcel, setTrackedParcel] = useState<Parcel | null | undefined>(undefined);
  const [trackingId, setTrackingId] = useState('');
  const [isTracking, setIsTracking] = useState(false);
  const [showAnnouncement, setShowAnnouncement] = useState(true);

  useEffect(() => {
    const sellerParcels = mockParcels.filter(p => p.sellerId === mockUser.id);
    setParcels(sellerParcels);
    const sellerComplaints = mockComplaints.filter(c => c.sellerId === mockUser.id);
    setComplaints(sellerComplaints);
  }, []);

  const handleBookingComplete = (result: BookingResult) => {
    if (result) {
      setParcels(prevParcels => [result.parcel, ...prevParcels]);
      setBookingResult(result);
    }
  };

  const handleComplaintAdded = (newComplaint: Complaint) => {
    setComplaints(prevComplaints => [newComplaint, ...prevComplaints]);
  };

  const handleCloseEtaDialog = () => {
    setBookingResult(null);
  };
  
  const handleTrack = (id: string) => {
    setIsTracking(true);
    setTrackingId(id);
    const parcel = mockParcels.find(p => p.id.toLowerCase() === id.toLowerCase());
    setTimeout(() => {
      setTrackedParcel(parcel);
      setIsTracking(false);
    }, 1000);
  };

  const handleCloseTrackDialog = () => {
    setTrackedParcel(undefined);
    setTrackingId('');
  };

  return (
    <SidebarProvider>
      <DashboardLayout 
        onBookingComplete={handleBookingComplete} 
        bookParcelAction={bookParcelAction} 
        userRole='seller'
        showChildren={!showAnnouncement}
      >
        <div className="flex flex-col gap-6">
          
          <OverviewCards parcels={parcels} userRole="seller" />
          
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
             <BookParcelDialog onBookingComplete={handleBookingComplete} bookParcelAction={bookParcelAction}>
                <Button size="lg" variant="accent" className="w-full h-12 text-base">
                  <PlusCircle className="mr-2" />
                  Create New Shipment
                </Button>
              </BookParcelDialog>
              <NewTrackParcelDialog onTrack={handleTrack} isTracking={isTracking} onIdChange={setTrackingId} trackingId={trackingId}>
                  <Button size="lg" variant="default" className="w-full h-12 text-base">
                      <Search className="mr-2" />
                      Track Shipment
                  </Button>
              </NewTrackParcelDialog>
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6">
            <div className="xl:col-span-2">
                <ParcelsTable 
                  parcels={parcels.slice(0, 5)} 
                  showCardWrapper={true}
                  title="Recent Orders"
                  description="Your 5 most recent orders."
                  viewAllLink="/parcels"
                />
            </div>
            
            <div className="flex flex-col gap-6">
                <ComplaintsPreview complaints={complaints} onComplaintAdded={handleComplaintAdded} />
                <Card>
                  <CardHeader>
                    <CardTitle>Earnings</CardTitle>
                  </CardHeader>
                  <CardContent>
                    <DeliveryPerformanceChart parcels={parcels} />
                  </CardContent>
                  <CardFooter>
                    <Button variant="outline" className="w-full" asChild>
                          <Link href="/parcels">View All Orders</Link>
                    </Button>
                  </CardFooter>
                </Card>
            </div>
          </div>
        </div>
      </DashboardLayout>
       <ETAPredictionDialog
        result={bookingResult}
        open={!!bookingResult}
        onOpenChange={(isOpen) => !isOpen && handleCloseEtaDialog()}
      />
      <TrackParcelDialog
        parcel={trackedParcel}
        open={trackedParcel !== undefined}
        onOpenChange={(isOpen) => !isOpen && handleCloseTrackDialog()}
        trackingId={trackingId}
        isTracking={isTracking}
      />
      <AnnouncementDialog 
        open={showAnnouncement}
        onClose={() => setShowAnnouncement(false)}
      />
    </SidebarProvider>
  );
}
