
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, PackagingRequest } from '@/lib/types';
import { Card, CardContent, CardHeader, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Search, RefreshCw, Printer, Download, Settings, MoreVertical, UploadCloud } from 'lucide-react';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger, DropdownMenuSub, DropdownMenuSubTrigger, DropdownMenuSubContent, DropdownMenuRadioGroup, DropdownMenuRadioItem, DropdownMenuLabel } from '@/components/ui/dropdown-menu';

type LoadSheetLog = {
  id: string;
  loadsheetNumber: string;
  totalOrders: number;
  picked: number;
  unpicked: number;
  riderName: string;
  date: string;
  status: 'Complete' | 'In Progress';
};

const mockLoadSheetLogs: LoadSheetLog[] = [];

type SearchType = 'loadsheet' | 'rider';


export default function LoadSheetLogsPage() {
  
  const [logs, setLogs] = useState<LoadSheetLog[]>(mockLoadSheetLogs);
  const [searchType, setSearchType] = useState<SearchType>('loadsheet');
  const handleBookingComplete = (result: BookingResult) => {};

  const getStatusVariant = (status: LoadSheetLog['status']) => {
    switch (status) {
        case 'Complete': return 'bg-green-100 text-green-800';
        case 'In Progress': return 'bg-yellow-100 text-yellow-800';
        default: return 'bg-secondary';
    }
  }

  const [showCount, setShowCount] = useState("25");
  const [sortOrder, setSortOrder] = useState("desc");

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="seller">
        <div className="flex flex-col gap-8">
          
          <Card>
            <CardHeader>
                <div className="flex items-center justify-end gap-2">
                    <Button variant="outline" size="icon"><RefreshCw className="h-4 w-4" /></Button>
                    <Button variant="outline" size="icon"><UploadCloud className="h-4 w-4" /></Button>
                    <Button variant="outline" size="icon"><Download className="h-4 w-4" /></Button>
                    <Button variant="outline" size="icon"><Printer className="h-4 w-4" /></Button>
                    <DropdownMenu>
                        <DropdownMenuTrigger asChild>
                            <Button variant="outline" size="icon">
                                <Settings className="h-4 w-4" />
                            </Button>
                        </DropdownMenuTrigger>
                        <DropdownMenuContent align="end">
                            <DropdownMenuLabel>Show</DropdownMenuLabel>
                            <DropdownMenuRadioGroup value={showCount} onValueChange={setShowCount}>
                                <DropdownMenuRadioItem value="25">25</DropdownMenuRadioItem>
                                <DropdownMenuRadioItem value="50">50</DropdownMenuRadioItem>
                                <DropdownMenuRadioItem value="100">100</DropdownMenuRadioItem>
                                <DropdownMenuRadioItem value="200">200</DropdownMenuRadioItem>
                                <DropdownMenuRadioItem value="300">300</DropdownMenuRadioItem>
                                <DropdownMenuRadioItem value="all">All</DropdownMenuRadioItem>
                            </DropdownMenuRadioGroup>
                            <DropdownMenuLabel className="mt-2">Sort By Order Date</DropdownMenuLabel>
                            <DropdownMenuRadioGroup value={sortOrder} onValueChange={setSortOrder}>
                                <DropdownMenuRadioItem value="asc">ASC</DropdownMenuRadioItem>
                                <DropdownMenuRadioItem value="desc">DESC</DropdownMenuRadioItem>
                            </DropdownMenuRadioGroup>
                        </DropdownMenuContent>
                    </DropdownMenu>
                </div>
            </CardHeader>
            <CardContent className="space-y-6">
                <div className="p-4 border rounded-lg bg-muted/50">
                        <p className="text-sm font-semibold mb-4 text-muted-foreground">ADVANCED SEARCH</p>
                        <div className="flex flex-wrap items-end gap-4">
                            <div className="flex-1 min-w-[300px]">
                                <label className="text-xs text-muted-foreground">Search by</label>
                                <div className="flex">
                                    <Input placeholder="Search..." className="rounded-r-none" />
                                    <Button 
                                        variant={searchType === 'loadsheet' ? 'primary' : 'secondary'} 
                                        className="rounded-none border-l-0"
                                        onClick={() => setSearchType('loadsheet')}
                                    >
                                        LOADSHEET #
                                    </Button>
                                    <Button 
                                        variant={searchType === 'rider' ? 'primary' : 'secondary'} 
                                        className="rounded-l-none"
                                        onClick={() => setSearchType('rider')}
                                    >
                                        RIDER
                                    </Button>
                                </div>
                            </div>
                            <div className="flex items-center gap-2">
                                <Button><Search className="mr-2 h-4 w-4"/> Search</Button>
                                <Button variant="secondary">Clear Filter</Button>
                            </div>
                        </div>
                </div>

              <div className="overflow-x-auto">
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>LOADSHEET #</TableHead>
                                <TableHead>TOTAL ORDERS</TableHead>
                                <TableHead>PICKED</TableHead>
                                <TableHead>UNPICKED</TableHead>
                                <TableHead>RIDER NAME</TableHead>
                                <TableHead>DATE</TableHead>
                                <TableHead>STATUS</TableHead>
                                <TableHead>ACTION</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                        {logs.length === 0 ? (
                            <TableRow>
                                <TableCell colSpan={8} className="h-24 text-center">No load sheet logs found.</TableCell>
                            </TableRow>
                        ) : logs.map((log) => (
                            <TableRow key={log.id}>
                                <TableCell className="font-medium text-primary">{log.loadsheetNumber}</TableCell>
                                <TableCell>{log.totalOrders}</TableCell>
                                <TableCell>{log.picked}</TableCell>
                                <TableCell>{log.unpicked}</TableCell>
                                <TableCell>{log.riderName}</TableCell>
                                <TableCell>{log.date}</TableCell>
                                <TableCell>
                                    <Badge variant="outline" className={`${getStatusVariant(log.status)} capitalize`}>
                                        {log.status}
                                    </Badge>
                                </TableCell>
                                <TableCell>
                                     <DropdownMenu>
                                        <DropdownMenuTrigger asChild>
                                            <Button variant="ghost" size="icon" className="h-8 w-8">
                                                <MoreVertical className="h-4 w-4" />
                                            </Button>
                                        </DropdownMenuTrigger>
                                        <DropdownMenuContent align="end">
                                            <DropdownMenuItem>View Details</DropdownMenuItem>
                                            <DropdownMenuItem>Print Loadsheet</DropdownMenuItem>
                                        </DropdownMenuContent>
                                    </DropdownMenu>
                                </TableCell>
                            </TableRow>
                        ))}
                        </TableBody>
                    </Table>
                </div>
            </CardContent>
            <CardFooter>
                 <p className="text-sm text-muted-foreground">Showing {logs.length} of {logs.length} entries</p>
            </CardFooter>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
