
'use client';

import { useMemo } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { mockParcels, mockUser } from '@/lib/data';
import { format, startOfWeek, startOfMonth } from 'date-fns';
import { BarChart, Bar, XAxis, YAxis, CartesianGrid, Tooltip, Legend, ResponsiveContainer } from 'recharts';
import { ChartContainer, ChartTooltipContent } from '@/components/ui/chart';


export default function AmountWisePage() {
  
  const handleBookingComplete = (result: BookingResult) => {};

  const amountData = useMemo(() => {
    const sellerParcels = mockParcels.filter(p => p.sellerId === mockUser.id);
    
    const weekly = sellerParcels.reduce((acc, p) => {
        const weekStart = format(startOfWeek(new Date(p.bookingDate)), 'yyyy-MM-dd');
        acc[weekStart] = (acc[weekStart] || 0) + p.codAmount;
        return acc;
    }, {} as Record<string, number>);

    const monthly = sellerParcels.reduce((acc, p) => {
        const monthStart = format(startOfMonth(new Date(p.bookingDate)), 'MMM yyyy');
        acc[monthStart] = (acc[monthStart] || 0) + p.codAmount;
        return acc;
    }, {} as Record<string, number>);

    return {
        weekly: Object.entries(weekly).map(([week, amount]) => ({ name: week, amount: Math.round(amount) })).sort((a, b) => a.name.localeCompare(b.name)),
        monthly: Object.entries(monthly).map(([month, amount]) => ({ name: month, amount: Math.round(amount) })).sort((a, b) => new Date(a.name) > new Date(b.name) ? 1 : -1)
    };
  }, []);


  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="seller">
        <div className="flex flex-col gap-8">
          
          <div className="grid gap-8 lg:grid-cols-2">
            <Card>
              <CardHeader>
                <CardTitle>Weekly Order Amount (PKR)</CardTitle>
                <CardDescription>
                  Total COD amount from orders per week.
                </CardDescription>
              </CardHeader>
              <CardContent>
                <ChartContainer config={{}} className="h-80 w-full">
                    <ResponsiveContainer>
                        <BarChart data={amountData.weekly}>
                            <CartesianGrid strokeDasharray="3 3" />
                            <XAxis dataKey="name" />
                            <YAxis />
                            <Tooltip content={<ChartTooltipContent formatter={(value) => `PKR ${value}`} />} />
                            <Legend />
                            <Bar dataKey="amount" fill="hsl(var(--primary))" />
                        </BarChart>
                    </ResponsiveContainer>
                </ChartContainer>
              </CardContent>
            </Card>
             <Card>
              <CardHeader>
                <CardTitle>Monthly Order Amount (PKR)</CardTitle>
                <CardDescription>
                  Total COD amount from orders per month.
                </CardDescription>
              </CardHeader>
              <CardContent>
                <ChartContainer config={{}} className="h-80 w-full">
                    <ResponsiveContainer>
                        <BarChart data={amountData.monthly}>
                            <CartesianGrid strokeDasharray="3 3" />
                            <XAxis dataKey="name" />
                            <YAxis />
                            <Tooltip content={<ChartTooltipContent formatter={(value) => `PKR ${value}`} />} />
                            <Legend />
                            <Bar dataKey="amount" fill="hsl(var(--accent))" />
                        </BarChart>
                    </ResponsiveContainer>
                </ChartContainer>
              </CardContent>
            </Card>
          </div>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
