
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel } from '@/lib/types';
import { Card, CardContent, CardHeader, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { DatePicker } from '@/components/ui/date-picker';
import { Search, RefreshCw, Printer, Download, Settings, Trash2, Pencil, Copy, PlusCircle, UploadCloud } from 'lucide-react';
import { format } from 'date-fns';
import { BookParcelDialog } from '@/components/book-parcel-dialog';
import { bookParcelAction } from '@/app/dashboard/actions';
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuItem, DropdownMenuSub, DropdownMenuSubTrigger, DropdownMenuSubContent, DropdownMenuRadioGroup, DropdownMenuRadioItem, DropdownMenuLabel } from '@/components/ui/dropdown-menu';


type SearchType = 'tracking' | 'product' | 'name' | 'phone';

export default function OrderLogsPage() {
    const [parcels, setParcels] = useState<Parcel[]>([]);
    const [searchType, setSearchType] = useState<SearchType>('tracking');
    const [selectedParcels, setSelectedParcels] = useState<string[]>([]);
    const [bookingResult, setBookingResult] = useState<BookingResult>(null);

    const handleBookingComplete = (result: BookingResult) => {
        if (result) {
            setParcels(prev => [result.parcel, ...prev]);
            setBookingResult(result);
        }
    };
    
    const handleSelectAll = (checked: boolean) => {
        if (checked) {
            setSelectedParcels(parcels.map(p => p.id));
        } else {
            setSelectedParcels([]);
        }
    };

    const handleSelectRow = (parcelId: string, checked: boolean) => {
        if (checked) {
            setSelectedParcels(prev => [...prev, parcelId]);
        } else {
            setSelectedParcels(prev => prev.filter(id => id !== parcelId));
        }
    };

    const getStatusVariant = (status: Parcel['status']) => {
        switch (status) {
            case 'Delivered': return 'bg-green-100 text-green-800';
            case 'Returned':
            case 'Failed':
                return 'bg-red-100 text-red-800';
            case 'In Transit':
                return 'bg-blue-100 text-blue-800';
            default:
                return 'bg-yellow-100 text-yellow-800';
        }
    };
    
    const [showCount, setShowCount] = useState("25");
    const [sortOrder, setSortOrder] = useState("desc");

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="seller" bookParcelAction={bookParcelAction}>
        <div className="flex flex-col gap-8">
            <div className="flex items-center justify-end">
                <BookParcelDialog onBookingComplete={handleBookingComplete} bookParcelAction={bookParcelAction}>
                    <Button>
                        <PlusCircle className="mr-2 h-4 w-4" />
                        Create Order
                    </Button>
                </BookParcelDialog>
            </div>
            <Card>
                <CardHeader className="p-4">
                    <div className="flex flex-wrap items-center justify-between gap-4">
                        <div className="flex items-center gap-1 border-b flex-wrap">
                            <Button variant="ghost" className="border-b-2 border-primary text-primary rounded-none text-xs h-8">All (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Booked (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Unbooked (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">In-Transit (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Delivered (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Returned (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Cancelled (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Regret (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Lost (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Stolen (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none text-xs h-8">Damage (0)</Button>
                        </div>
                        <div className="flex items-center gap-2">
                            <Button variant="outline" size="icon"><RefreshCw className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><UploadCloud className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><Download className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><Printer className="h-4 w-4" /></Button>
                            <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <Button variant="outline" size="icon">
                                        <Settings className="h-4 w-4" />
                                    </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                    <DropdownMenuLabel>Show</DropdownMenuLabel>
                                    <DropdownMenuRadioGroup value={showCount} onValueChange={setShowCount}>
                                        <DropdownMenuRadioItem value="25">25</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="50">50</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="100">100</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="200">200</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="300">300</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="all">All</DropdownMenuRadioItem>
                                    </DropdownMenuRadioGroup>
                                    <DropdownMenuLabel className="mt-2">Sort By Order Date</DropdownMenuLabel>
                                    <DropdownMenuRadioGroup value={sortOrder} onValueChange={setSortOrder}>
                                        <DropdownMenuRadioItem value="asc">ASC</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="desc">DESC</DropdownMenuRadioItem>
                                    </DropdownMenuRadioGroup>
                                </DropdownMenuContent>
                            </DropdownMenu>
                        </div>
                    </div>
                </CardHeader>
                <CardContent className="space-y-6">
                    <div className="p-4 border rounded-lg bg-muted/50">
                        <p className="text-sm font-semibold mb-4">ADVANCED SEARCH</p>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                            <div className="lg:col-span-2">
                                <label className="text-xs text-muted-foreground">Search by</label>
                                <div className="flex">
                                    <Input placeholder="Search..." className="rounded-r-none" />
                                    <Select value={searchType} onValueChange={(val) => setSearchType(val as SearchType)}>
                                        <SelectTrigger className="w-auto min-w-[160px] rounded-l-none border-l-0">
                                            <SelectValue />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="tracking">TRACKING #</SelectItem>
                                            <SelectItem value="product">PRODUCT ID</SelectItem>
                                            <SelectItem value="name">NAME</SelectItem>
                                            <SelectItem value="phone">PHONE #</SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>
                            </div>
                            <div>
                                <label className="text-xs text-muted-foreground">FROM</label>
                                <DatePicker />
                            </div>
                             <div>
                                <label className="text-xs text-muted-foreground">TO</label>
                                <DatePicker />
                            </div>
                             <div>
                                <label className="text-xs text-muted-foreground">STATUS</label>
                                <Select><SelectTrigger><SelectValue placeholder="Select Status" /></SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="Pending">Pending</SelectItem>
                                    <SelectItem value="Picked Up">Picked Up</SelectItem>
                                    <SelectItem value="In Transit">In Transit</SelectItem>
                                    <SelectItem value="Delivered">Delivered</SelectItem>
                                    <SelectItem value="Returned">Returned</SelectItem>
                                    <SelectItem value="Failed">Failed</SelectItem>
                                    <SelectItem value="Unbooked">Unbooked</SelectItem>
                                    <SelectItem value="In Warehouse">In Warehouse</SelectItem>
                                    <SelectItem value="Out for Delivery">Out for Delivery</SelectItem>
                                    <SelectItem value="Attempted">Attempted</SelectItem>
                                    <SelectItem value="Out for Return">Out for Return</SelectItem>
                                    <SelectItem value="Shipper Advice">Shipper Advice</SelectItem>
                                    <SelectItem value="Branch Shift">Branch Shift</SelectItem>
                                    <SelectItem value="Cancelled">Cancelled</SelectItem>
                                    <SelectItem value="Lost">Lost</SelectItem>
                                    <SelectItem value="Stolen">Stolen</SelectItem>
                                    <SelectItem value="Damaged">Damaged</SelectItem>
                                </SelectContent>
                                </Select>
                            </div>
                             <div>
                                <label className="text-xs text-muted-foreground">ORDER DATE TYPE</label>
                                <Select defaultValue="transaction-date">
                                    <SelectTrigger><SelectValue /></SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="transaction-date">Transaction Date</SelectItem>
                                        <SelectItem value="delivery-date">Delivery Date</SelectItem>
                                        <SelectItem value="pickup-date">PickUp Date</SelectItem>
                                        <SelectItem value="stock-in-picked-at">Stock In Picked At</SelectItem>
                                        <SelectItem value="status-changed-at">Order Status Changed At</SelectItem>
                                        <SelectItem value="booked-at">Booked At</SelectItem>
                                    </SelectContent>
                                </Select>
                            </div>
                             <div>
                                <label className="text-xs text-muted-foreground">PICKUP ADDRESS</label>
                                <Select><SelectTrigger><SelectValue placeholder="Search" /></SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="Main Office">Main Office</SelectItem>
                                    <SelectItem value="Warehouse A">Warehouse A</SelectItem>
                                </SelectContent>
                                </Select>
                            </div>
                            <div className="flex items-end gap-2">
                                <Button className="w-full"><Search className="mr-2 h-4 w-4"/> Search</Button>
                                <Button variant="secondary" className="w-full">Clear Filter</Button>
                            </div>
                        </div>
                    </div>

                    <div className="overflow-x-auto">
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead><Checkbox onCheckedChange={(c) => handleSelectAll(c as boolean)} checked={selectedParcels.length > 0 && selectedParcels.length === parcels.length} /></TableHead>
                                    <TableHead>ORDER REF</TableHead>
                                    <TableHead>TRACKING #</TableHead>
                                    <TableHead>SHIPPER</TableHead>
                                    <TableHead>WEIGHT</TableHead>
                                    <TableHead>DELIVERY ADDRESS</TableHead>
                                    <TableHead>RETURN ADDRESS</TableHead>
                                    <TableHead>PIECE</TableHead>
                                    <TableHead>DETAILS</TableHead>
                                    <TableHead className="text-right">AMOUNT</TableHead>
                                    <TableHead>JOURNEY</TableHead>
                                    <TableHead>STATUS</TableHead>
                                    <TableHead>ACTION</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {parcels.length === 0 ? (
                                    <TableRow>
                                        <TableCell colSpan={13} className="h-24 text-center">No orders found.</TableCell>
                                    </TableRow>
                                ) : parcels.map(parcel => (
                                    <TableRow key={parcel.id} data-state={selectedParcels.includes(parcel.id) && "selected"}>
                                        <TableCell><Checkbox checked={selectedParcels.includes(parcel.id)} onCheckedChange={(c) => handleSelectRow(parcel.id, c as boolean)} /></TableCell>
                                        <TableCell>
                                            <div className="font-medium">{parcel.orderType}</div>
                                            <div className="text-xs text-muted-foreground">{format(new Date(parcel.bookingDate), 'dd-MMM-yyyy')}</div>
                                        </TableCell>
                                        <TableCell>
                                            <Button variant="link" className="p-0 h-auto text-primary">{parcel.orderReferenceNumber}</Button>
                                        </TableCell>
                                        <TableCell>
                                            <div className="font-medium">{parcel.recipientName}</div>
                                            <div className="text-xs text-muted-foreground">{parcel.contact}</div>
                                        </TableCell>
                                        <TableCell>{parcel.bookingWeight}kg</TableCell>
                                        <TableCell>
                                            <div className="w-40 truncate">{parcel.address}</div>
                                            <Badge variant="secondary">{parcel.city}</Badge>
                                        </TableCell>
                                        <TableCell>
                                            <div className="w-40 truncate">{parcel.returnAddress ?? 'N/A'}</div>
                                            <Badge variant="secondary">{parcel.returnCity ?? 'N/A'}</Badge>
                                        </TableCell>
                                        <TableCell>{parcel.items}</TableCell>
                                        <TableCell>{parcel.orderDetail}</TableCell>
                                        <TableCell className="text-right">PKR {parcel.codAmount.toFixed(2)}</TableCell>
                                        <TableCell><Badge variant="outline" className="bg-green-100 text-green-800">Forward</Badge></TableCell>
                                        <TableCell>
                                            <Badge variant="outline" className={`${getStatusVariant(parcel.status)}`}>{parcel.status}</Badge>
                                            <div className="text-xs text-muted-foreground mt-1">Re-route to {parcel.city} warehouse</div>
                                        </TableCell>
                                        <TableCell>
                                            <div className="flex items-center gap-1">
                                                <Button variant="ghost" size="icon" className="h-7 w-7"><Copy className="h-4 w-4"/></Button>
                                                <Button variant="ghost" size="icon" className="h-7 w-7"><Pencil className="h-4 w-4"/></Button>
                                                <Button variant="ghost" size="icon" className="h-7 w-7 text-red-500 hover:text-red-700"><Trash2 className="h-4 w-4" /></Button>
                                            </div>
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </div>
                </CardContent>
                <CardFooter className="flex-col items-start gap-4 sm:flex-row sm:items-center sm:justify-between">
                    <div className="text-sm text-muted-foreground">
                        Showing <strong>{selectedParcels.length}</strong> of <strong>{parcels.length}</strong> entries
                    </div>
                    <div className="flex items-center gap-2">
                        <Select defaultValue="1">
                            <SelectTrigger className="w-16"><SelectValue /></SelectTrigger>
                            <SelectContent>
                                <SelectItem value="1">1</SelectItem>
                            </SelectContent>
                        </Select>
                        <span className="text-sm">of 1</span>
                        <Button variant="outline" size="sm" disabled>Previous</Button>
                        <Button variant="outline" size="sm" disabled>Next</Button>
                    </div>
                </CardFooter>
            </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}

    

    

    

    