'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, PackagingLog } from '@/lib/types';
import { Card, CardContent, CardHeader, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';
import { Search, RefreshCw, Printer, Download, Settings, PlusCircle, LayoutGrid } from 'lucide-react';
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuTrigger, DropdownMenuRadioGroup, DropdownMenuRadioItem, DropdownMenuLabel } from '@/components/ui/dropdown-menu';
import { CreatePackagingRequestDialog } from '@/components/packaging/create-request-dialog';
import { useToast } from '@/hooks/use-toast';
import { DatePicker } from '@/components/ui/date-picker';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';

type SearchType = 'package' | 'price';

const statusStyles: Record<PackagingLog['status'], string> = {
  Pending: 'bg-yellow-100 text-yellow-800',
  Approved: 'bg-green-100 text-green-800',
  'Un-Assigned': 'bg-gray-200 text-gray-800',
};

export default function PackagingLogsPage() {
  const { toast } = useToast();
  const [requests, setRequests] = useState<PackagingLog[]>([]);
  const [searchType, setSearchType] = useState<SearchType>('package');

  const handleRequestCreated = (newRequest: any) => { // TODO: fix any type
    toast({
        title: 'Request Submitted!',
        description: `Your packaging request for ${newRequest.quantity} items has been logged.`,
    });
    // In a real app, you'd likely re-fetch the data here.
    // For now, we'll just add it to the local state.
    // setRequests(prev => [newRequest, ...prev]);
  };

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={() => {}} userRole="seller">
        <div className="flex flex-col gap-8">
            <div className="flex items-center justify-between">
                <h1 className="text-2xl font-semibold">Packaging Logs</h1>
                <CreatePackagingRequestDialog onRequestCreated={handleRequestCreated}>
                    <Button>
                        <PlusCircle className="mr-2 h-4 w-4" />
                        Add New
                    </Button>
                </CreatePackagingRequestDialog>
            </div>
          <Card>
            <CardHeader className="p-4 border-b">
               <div className="flex flex-wrap items-center justify-between gap-4">
                    <div className="flex items-center gap-2 border-b">
                        <Button variant="ghost" className="border-b-2 border-primary text-primary rounded-none">All ({requests.length})</Button>
                        <Button variant="ghost" className="text-muted-foreground rounded-none">New ({requests.filter(r => r.status === 'Pending').length})</Button>
                        <Button variant="ghost" className="text-muted-foreground rounded-none">Approved ({requests.filter(r => r.status === 'Approved').length})</Button>
                        <Button variant="ghost" className="text-muted-foreground rounded-none">Un-Assigned By Me ({requests.filter(r => r.status === 'Un-Assigned').length})</Button>
                    </div>
                    <div className="flex items-center gap-2">
                        <Button variant="outline" size="icon"><RefreshCw className="h-4 w-4" /></Button>
                        <Button variant="outline" size="icon"><LayoutGrid className="h-4 w-4" /></Button>
                        <Button variant="outline" size="icon"><Download className="h-4 w-4" /></Button>
                        <Button variant="outline" size="icon"><Settings className="h-4 w-4" /></Button>
                    </div>
                </div>
            </CardHeader>
            <CardContent className="space-y-6 pt-6">
               <div className="p-4 border rounded-lg bg-muted/50">
                    <p className="text-sm font-semibold mb-4 text-muted-foreground">ADVANCED SEARCH</p>
                    <div className="grid grid-cols-1 md:grid-cols-3 lg:grid-cols-4 gap-4 items-end">
                        <div className="md:col-span-2">
                            <label className="text-xs text-muted-foreground">Search by</label>
                            <div className="flex">
                                <Input placeholder="Search..." className="rounded-r-none" />
                                 <Button 
                                    variant={searchType === 'package' ? 'secondary' : 'outline'} 
                                    className="rounded-none border-l-0"
                                    onClick={() => setSearchType('package')}
                                >
                                    PACKAGE #
                                </Button>
                                <Button 
                                    variant={searchType === 'price' ? 'secondary' : 'outline'} 
                                    className="rounded-l-none"
                                    onClick={() => setSearchType('price')}
                                >
                                    PRICE
                                </Button>
                            </div>
                        </div>
                        <div>
                            <label className="text-xs text-muted-foreground">FROM</label>
                            <DatePicker />
                        </div>
                        <div>
                            <label className="text-xs text-muted-foreground">TO</label>
                            <DatePicker />
                        </div>
                         <div className="flex items-end gap-2 col-start-1">
                            <Button><Search className="mr-2 h-4 w-4"/> Search</Button>
                            <Button variant="secondary">Clear Filter</Button>
                        </div>
                    </div>
                </div>
                 <div className="overflow-x-auto">
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>PACKAGE #</TableHead>
                                <TableHead>PRICE</TableHead>
                                <TableHead>MERCHANT NAME</TableHead>
                                <TableHead>ADDRESS</TableHead>
                                <TableHead>CITY</TableHead>
                                <TableHead>PACKAGE DATE</TableHead>
                                <TableHead>DESCRIPTION</TableHead>
                                <TableHead>STATUS</TableHead>
                                <TableHead>ACTION</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                        {requests.length === 0 ? (
                            <TableRow>
                                <TableCell colSpan={9} className="h-24 text-center">No packaging logs found.</TableCell>
                            </TableRow>
                        ) : requests.map((request) => (
                            <TableRow key={request.id}>
                                <TableCell className="font-medium">{request.packageNumber}</TableCell>
                                <TableCell>{request.price.toFixed(2)}</TableCell>
                                <TableCell>{request.merchantName}</TableCell>
                                <TableCell className="w-48 truncate">{request.address}</TableCell>
                                <TableCell>{request.city}</TableCell>
                                <TableCell>{request.packageDate}</TableCell>
                                <TableCell className="w-48 truncate">{request.description}</TableCell>
                                <TableCell>
                                    <Badge variant="outline" className={statusStyles[request.status]}>
                                        {request.status}
                                    </Badge>
                                </TableCell>
                                <TableCell>{/* Action Buttons */}</TableCell>
                            </TableRow>
                        ))}
                        </TableBody>
                    </Table>
                </div>
            </CardContent>
             <CardFooter className="flex-col items-start gap-4 sm:flex-row sm:items-center sm:justify-between pt-4 border-t">
                <p className="text-sm text-muted-foreground">Showing {requests.length} of {requests.length} entries</p>
                <div className="flex items-center gap-2">
                    <span>Page</span>
                    <Select defaultValue="1">
                        <SelectTrigger className="w-16"><SelectValue /></SelectTrigger>
                        <SelectContent>
                            <SelectItem value="1">1</SelectItem>
                        </SelectContent>
                    </Select>
                    <span className="text-sm">of 1</span>
                    <Button variant="outline" size="sm" disabled>Previous</Button>
                    <Button variant="outline" size="sm" disabled>Next</Button>
                </div>
            </CardFooter>
          </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
