
'use client';

import { useState } from 'react';
import { SwiftSendLogo } from '@/components/icons';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { ArrowRight, Package, ShipWheel, BarChart, CreditCard, Zap, Shield } from 'lucide-react';
import Link from 'next/link';
import Image from 'next/image';
import { TrackParcelDialog } from '@/components/track-parcel-dialog';
import type { Parcel } from '@/lib/types';
import { mockParcels } from '@/lib/data';
import { cn } from '@/lib/utils';
import { Badge } from '@/components/ui/badge';

const FeatureCard = ({ icon: Icon, title, description }: { icon: React.ElementType, title: string, description: string }) => (
    <div className="flex flex-col items-center p-6 text-center bg-card rounded-xl shadow-sm hover:shadow-lg transition-shadow">
        <div className="flex h-12 w-12 items-center justify-center rounded-full bg-primary/10 text-primary mb-4">
            <Icon className="h-6 w-6" />
        </div>
        <h3 className="text-lg font-semibold mb-2">{title}</h3>
        <p className="text-sm text-muted-foreground">{description}</p>
    </div>
);


export default function LandingPage() {
    const [trackingId, setTrackingId] = useState('');
    const [trackedParcel, setTrackedParcel] = useState<Parcel | null | undefined>(undefined);
    const [isTracking, setIsTracking] = useState(false);

    const handleTrack = () => {
        setIsTracking(true);
        // Mock finding the parcel
        const parcel = mockParcels.find(p => p.id.toLowerCase() === trackingId.toLowerCase());
        setTimeout(() => {
            setTrackedParcel(parcel);
            setIsTracking(false);
        }, 1000);
    };
    
    const handleCloseDialog = () => {
        setTrackedParcel(undefined);
        setTrackingId('');
    };

    return (
        <div className="flex flex-col min-h-screen bg-background">
            <header className="sticky top-0 z-50 w-full border-b bg-background/95 backdrop-blur-sm">
                <div className="container mx-auto flex h-16 items-center justify-between px-4 md:px-6">
                    <Link href="/" className="flex items-center gap-2">
                        <SwiftSendLogo className="h-7 w-7 text-primary" />
                        <span className="text-xl font-bold">Al-wasi Soft Tech</span>
                    </Link>
                    <nav className="hidden items-center gap-6 md:flex">
                        <Link href="#features" className="text-sm font-medium text-muted-foreground transition-colors hover:text-primary">Features</Link>
                        <Link href="#how-it-works" className="text-sm font-medium text-muted-foreground transition-colors hover:text-primary">How It Works</Link>
                        <Link href="/pricing" className="text-sm font-medium text-muted-foreground transition-colors hover:text-primary">Pricing</Link>
                    </nav>
                    <div className="flex items-center gap-2">
                        <Button variant="ghost" asChild>
                            <Link href="/login">Log In</Link>
                        </Button>
                        <Button asChild>
                            <Link href="/login">Sign Up <ArrowRight className="ml-2" /></Link>
                        </Button>
                    </div>
                </div>
            </header>

            <main className="flex-1">
                <section className="container mx-auto px-4 md:px-6 py-20 md:py-32 text-center">
                    <Badge variant="outline" className="mb-4">
                        End-to-End COD Solution
                    </Badge>
                    <h1 className="text-4xl md:text-6xl font-bold tracking-tight">
                        Reliable Cash on Delivery <br /> for Your E-Commerce Business
                    </h1>
                    <p className="mt-6 max-w-2xl mx-auto text-lg text-muted-foreground">
                        Al-wasi Soft Tech simplifies your logistics with fast, secure, and transparent cash on delivery services. Focus on growing your business; we'll handle the deliveries.
                    </p>
                    <div className="mt-8 flex flex-col sm:flex-row items-center justify-center gap-4 max-w-lg mx-auto">
                        <Input 
                            type="text" 
                            placeholder="Enter your tracking ID" 
                            className="h-12 text-base text-center sm:text-left"
                            value={trackingId}
                            onChange={(e) => setTrackingId(e.target.value)}
                        />

                        <Button 
                            size="lg" 
                            className="w-full sm:w-auto h-12"
                            onClick={handleTrack}
                            disabled={!trackingId || isTracking}
                        >
                            {isTracking ? 'Tracking...' : 'Track Parcel'}
                        </Button>
                    </div>
                </section>
                
                <section id="features" className="bg-muted/50 py-20 md:py-24">
                    <div className="container mx-auto px-4 md:px-6">
                        <div className="text-center">
                            <h2 className="text-3xl font-bold">Everything You Need to Succeed</h2>
                            <p className="mt-4 max-w-xl mx-auto text-muted-foreground">
                                From booking to reconciliation, our platform provides all the tools for a seamless delivery experience.
                            </p>
                        </div>
                        <div className="mt-12 grid gap-8 md:grid-cols-2 lg:grid-cols-3">
                            <FeatureCard 
                                icon={Package}
                                title="Effortless Booking"
                                description="Book single or bulk parcels in seconds with our intuitive interface and API integrations."
                            />
                            <FeatureCard 
                                icon={ShipWheel}
                                title="Real-Time Tracking"
                                description="Keep your customers informed with live tracking updates from pickup to final delivery."
                            />
                            <FeatureCard 
                                icon={CreditCard}
                                title="Automated COD Reconciliation"
                                description="Our system automates cash on delivery settlements, ensuring you get paid faster."
                            />
                            <FeatureCard 
                                icon={BarChart}
                                title="Insightful Analytics"
                                description="Access detailed reports on delivery performance, COD collections, and rider efficiency."
                            />
                            <FeatureCard 
                                icon={Zap}
                                title="Powerful Integrations"
                                description="Connect your Shopify store or other e-commerce platforms to automate order fulfillment."
                            />
                            <FeatureCard 
                                icon={Shield}
                                title="Secure & Reliable"
                                description="With role-based access and activity logs, your data and operations are always secure."
                            />
                        </div>
                    </div>
                </section>

                <section id="how-it-works" className="py-20 md:py-24">
                    <div className="container mx-auto px-4 md:px-6">
                        <div className="text-center">
                            <h2 className="text-3xl font-bold">Get Started in 3 Simple Steps</h2>
                            <p className="mt-4 max-w-xl mx-auto text-muted-foreground">
                                Our streamlined process makes courier management a breeze.
                            </p>
                        </div>
                        <div className="relative mt-16 grid gap-8 md:grid-cols-3">
                             <div className="absolute top-1/2 left-0 w-full h-0.5 bg-border -translate-y-1/2 hidden md:block"></div>
                             <div className="relative flex flex-col items-center text-center">
                                 <div className="flex items-center justify-center h-16 w-16 rounded-full bg-primary text-primary-foreground border-4 border-background mb-4 text-xl font-bold">1</div>
                                 <h3 className="text-lg font-semibold mb-2">Book Your Parcel</h3>
                                 <p className="text-sm text-muted-foreground">Create a seller account and book your parcels through our dashboard or API.</p>
                             </div>
                             <div className="relative flex flex-col items-center text-center">
                                <div className="flex items-center justify-center h-16 w-16 rounded-full bg-primary text-primary-foreground border-4 border-background mb-4 text-xl font-bold">2</div>
                                <h3 className="text-lg font-semibold mb-2">We Pick Up & Deliver</h3>
                                <p className="text-sm text-muted-foreground">Our rider picks up the parcel and delivers it to your customer, collecting the COD amount.</p>
                            </div>
                            <div className="relative flex flex-col items-center text-center">
                                <div className="flex items-center justify-center h-16 w-16 rounded-full bg-primary text-primary-foreground border-4 border-background mb-4 text-xl font-bold">3</div>
                                <h3 className="text-lg font-semibold mb-2">Receive Your Payment</h3>
                                <p className="text-sm text-muted-foreground">The collected amount is credited to your wallet, ready for you to request a payout.</p>
                            </div>
                        </div>
                    </div>
                </section>

                <section className="bg-primary/5 border-t">
                    <div className="container mx-auto px-4 md:px-6 py-16 text-center">
                        <h2 className="text-3xl font-bold text-primary-foreground">Ready to streamline your deliveries?</h2>
                        <p className="mt-4 text-lg text-primary-foreground/80">Join hundreds of businesses who trust Al-wasi Soft Tech for their COD logistics.</p>
                        <Button size="lg" className="mt-8" asChild>
                             <Link href="/login">Get Started for Free <ArrowRight className="ml-2" /></Link>
                        </Button>
                    </div>
                </section>
            </main>

             <TrackParcelDialog
                parcel={trackedParcel}
                open={trackedParcel !== undefined}
                onOpenChange={(isOpen) => !isOpen && handleCloseDialog()}
                trackingId={trackingId}
             />
        </div>
    );
}
