
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, Parcel } from '@/lib/types';
import { Card, CardContent, CardHeader, CardFooter, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { DatePicker } from '@/components/ui/date-picker';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Badge } from '@/components/ui/badge';
import { Search, RefreshCw, Printer, Download, Settings, LayoutGrid, MoreVertical, UploadCloud } from 'lucide-react';
import { format } from 'date-fns';
import { DropdownMenu, DropdownMenuTrigger, DropdownMenuContent, DropdownMenuRadioGroup, DropdownMenuRadioItem, DropdownMenuLabel } from '@/components/ui/dropdown-menu';
import { mockParcels } from '@/lib/data';

type SearchType = 'order' | 'tracking' | 'name' | 'phone';

const allStatuses = [
    "Unbooked",
    "Booked",
    "PostEx Warehouse",
    "Out For Delivery",
    "Delivered",
    "Returned",
    "Un-Assigned By Me",
    "Expired",
    "Delivery Under Review",
    "Picked By PostEx",
    "Out For Return",
    "Attempted",
    "Transferred",
];

const getStatusBadge = (status: Parcel['status']) => {
    switch(status) {
        case 'Booked':
            return <Badge variant="outline">Booked</Badge>;
        case 'In Transit':
             return (
                <div className="flex flex-col items-start gap-1">
                    <Badge variant="secondary" className="w-full justify-center">En-Route to Lahore warehouse</Badge>
                    <Badge variant="default" className="bg-blue-500 hover:bg-blue-600 w-full justify-center">In-Transit</Badge>
                </div>
            );
        case 'Pending':
            return (
                <div className="flex flex-col items-start gap-1">
                    <Badge variant="secondary" className="w-full justify-center">PostEx Warehouse</Badge>
                    <Badge variant="default" className="bg-blue-500 hover:bg-blue-600 w-full justify-center">In-Transit</Badge>
                </div>
            );
        default:
            return <Badge>{status}</Badge>;
    }
}

export default function PickupReportPage() {
  const [parcels] = useState<Parcel[]>(mockParcels.slice(0, 2));
  const [searchType, setSearchType] = useState<SearchType>('order');
  const [showCount, setShowCount] = useState("25");
  const [sortOrder, setSortOrder] = useState("desc");

  const handleBookingComplete = (result: BookingResult) => {};

  return (
    <SidebarProvider>
      <DashboardLayout onBookingComplete={handleBookingComplete} userRole="seller">
        <div className="flex flex-col gap-8">
            <h1 className="text-2xl font-bold">Pick up Report</h1>
            <Card>
                <CardHeader>
                    <div className="flex flex-wrap items-center justify-between gap-4">
                        <div className="flex items-center gap-2 border-b">
                            <Button variant="ghost" className="border-b-2 border-primary text-primary rounded-none">All ({parcels.length})</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none">InTransit (10)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none">Delivered (11)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none">Return In-Transit (0)</Button>
                            <Button variant="ghost" className="text-muted-foreground rounded-none">Returned (0)</Button>
                        </div>
                        <div className="flex items-center gap-2">
                            <Button variant="outline" size="icon"><RefreshCw className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><LayoutGrid className="h-4 w-4" /></Button>
                            <Button variant="outline" size="icon"><Download className="h-4 w-4" /></Button>
                            <DropdownMenu>
                                <DropdownMenuTrigger asChild>
                                    <Button variant="outline" size="icon">
                                        <Settings className="h-4 w-4" />
                                    </Button>
                                </DropdownMenuTrigger>
                                <DropdownMenuContent align="end">
                                    <DropdownMenuLabel>Show</DropdownMenuLabel>
                                    <DropdownMenuRadioGroup value={showCount} onValueChange={setShowCount}>
                                        <DropdownMenuRadioItem value="25">25</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="50">50</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="100">100</DropdownMenuRadioItem>
                                    </DropdownMenuRadioGroup>
                                    <DropdownMenuLabel>Sort By Order Date</DropdownMenuLabel>
                                     <DropdownMenuRadioGroup value={sortOrder} onValueChange={setSortOrder}>
                                        <DropdownMenuRadioItem value="asc">ASC</DropdownMenuRadioItem>
                                        <DropdownMenuRadioItem value="desc">DESC</DropdownMenuRadioItem>
                                    </DropdownMenuRadioGroup>
                                </DropdownMenuContent>
                            </DropdownMenu>
                        </div>
                    </div>
                </CardHeader>
                <CardContent className="space-y-6">
                    <div className="p-4 border rounded-lg bg-muted/50">
                        <p className="text-sm font-semibold mb-4 text-muted-foreground">ADVANCED SEARCH</p>
                        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4 items-end">
                            <div className="lg:col-span-2">
                                <label className="text-xs text-muted-foreground">Search by</label>
                                <div className="flex">
                                    <Input placeholder="Search by" className="rounded-r-none" />
                                     <Button 
                                        variant={searchType === 'order' ? 'secondary' : 'outline'}
                                        className="rounded-none border-l-0"
                                        onClick={() => setSearchType('order')}
                                    >ORDER #</Button>
                                     <Button 
                                        variant={searchType === 'tracking' ? 'secondary' : 'outline'}
                                        className="rounded-none border-l-0 border-r-0"
                                        onClick={() => setSearchType('tracking')}
                                    >TRACKING #</Button>
                                     <Button 
                                        variant={searchType === 'name' ? 'secondary' : 'outline'}
                                        className="rounded-none"
                                        onClick={() => setSearchType('name')}
                                    >NAME</Button>
                                    <Button 
                                        variant={searchType === 'phone' ? 'secondary' : 'outline'}
                                        className="rounded-l-none"
                                        onClick={() => setSearchType('phone')}
                                    >PHONE #</Button>
                                </div>
                            </div>
                            <div>
                                <label className="text-xs text-muted-foreground">FROM</label>
                                <DatePicker />
                            </div>
                            <div>
                                <label className="text-xs text-muted-foreground">TO</label>
                                <DatePicker />
                            </div>
                             <div>
                                <label className="text-xs text-muted-foreground">STATUS</label>
                                <Select>
                                    <SelectTrigger><SelectValue placeholder="Select Status"/></SelectTrigger>
                                    <SelectContent>
                                        <SelectItem value="All">All</SelectItem>
                                        {allStatuses.map(status => (
                                            <SelectItem key={status} value={status}>{status}</SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                            </div>
                             <div>
                                <label className="text-xs text-muted-foreground">PICKUP ADDRESS</label>
                                <Select>
                                    <SelectTrigger><SelectValue placeholder="Search"/></SelectTrigger>
                                    <SelectContent>
                                    </SelectContent>
                                </Select>
                            </div>
                             <div className="flex items-end gap-2">
                                <Button className="w-full md:w-auto"><Search className="mr-2 h-4 w-4"/> Search</Button>
                                <Button variant="secondary" className="w-full md:w-auto">Clear Filter</Button>
                            </div>
                        </div>
                    </div>

                    <div className="overflow-x-auto">
                        <Table>
                            <TableHeader>
                                <TableRow>
                                    <TableHead>PICKUP DATE</TableHead>
                                    <TableHead>ORDER REF</TableHead>
                                    <TableHead>TRACKING #</TableHead>
                                    <TableHead>CUSTOMER</TableHead>
                                    <TableHead>WEIGHT</TableHead>
                                    <TableHead>ADDRESS</TableHead>
                                    <TableHead>CITY</TableHead>
                                    <TableHead>DETAILS</TableHead>
                                    <TableHead className="text-right">AMOUNT</TableHead>
                                    <TableHead>STATUS</TableHead>
                                    <TableHead>ACTION</TableHead>
                                </TableRow>
                            </TableHeader>
                            <TableBody>
                                {parcels.length === 0 ? (
                                    <TableRow>
                                        <TableCell colSpan={11} className="h-24 text-center">No data to display.</TableCell>
                                    </TableRow>
                                ) : parcels.map(parcel => (
                                    <TableRow key={parcel.id}>
                                         <TableCell>
                                            <div className="font-medium">{format(new Date(parcel.bookingDate), 'yyyy-MM-dd')}</div>
                                        </TableCell>
                                        <TableCell>
                                            <div className="font-medium">6</div>
                                            <div className="text-muted-foreground text-xs">Normal</div>
                                        </TableCell>
                                        <TableCell>
                                            <div className="font-medium text-primary">{parcel.id}</div>
                                            <div className="text-muted-foreground text-xs">{format(new Date(parcel.bookingDate), 'yyyy-MM-dd')}</div>
                                        </TableCell>
                                        <TableCell>
                                            <div className="font-medium">{parcel.recipientName}</div>
                                            <div className="text-muted-foreground text-xs">{parcel.contact}</div>
                                        </TableCell>
                                        <TableCell>{parcel.bookingWeight ? `${parcel.bookingWeight} (kg)` : '-'}</TableCell>
                                        <TableCell className="w-48 truncate">{parcel.address}</TableCell>
                                        <TableCell>{parcel.city}</TableCell>
                                        <TableCell>{parcel.orderDetail ?? 'HAVE HAIR REGROWING SPRAY'}</TableCell>
                                        <TableCell className="text-right">{parcel.codAmount.toFixed(2)}</TableCell>
                                        <TableCell>{getStatusBadge(parcel.status)}</TableCell>
                                        <TableCell>
                                          <Button variant="ghost" size="icon" className="h-8 w-8">
                                            <MoreVertical className="h-4 w-4" />
                                          </Button>
                                        </TableCell>
                                    </TableRow>
                                ))}
                            </TableBody>
                        </Table>
                    </div>
                </CardContent>
                 <CardFooter className="flex-col items-start gap-4 sm:flex-row sm:items-center sm:justify-between">
                     <p className="text-sm text-muted-foreground">Showing {parcels.length} of {parcels.length} entries</p>
                </CardFooter>
            </Card>
        </div>
      </DashboardLayout>
    </SidebarProvider>
  );
}
