
'use client';

import { useState } from 'react';
import { SidebarProvider } from '@/components/ui/sidebar';
import { DashboardLayout } from '@/components/dashboard-layout';
import type { BookingResult, PackagingRequest } from '@/lib/types';
import { Card, CardContent, CardDescription, CardHeader, CardTitle, CardFooter } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { PlusCircle } from 'lucide-react';
import { CreatePackagingRequestDialog } from '@/components/packaging/create-request-dialog';
import { useToast } from '@/hooks/use-toast';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from '@/components/ui/table';
import { Badge } from '@/components/ui/badge';

const statusStyles: Record<PackagingRequest['status'], string> = {
  Pending: 'bg-yellow-100 text-yellow-800',
  Shipped: 'bg-blue-100 text-blue-800',
  Delivered: 'bg-green-100 text-green-800',
};


export default function RequestLogsPage() {
    const { toast } = useToast();
    const [requests, setRequests] = useState<PackagingRequest[]>([]);

    const handleRequestCreated = (newRequest: PackagingRequest) => {
        toast({
            title: 'Request Submitted!',
            description: `Your packaging request for ${newRequest.quantity} items has been logged.`,
        });
        setRequests(prev => [newRequest, ...prev]);
    };

    return (
        <SidebarProvider>
            <DashboardLayout onBookingComplete={() => {}} userRole="seller">
                <div className="flex flex-col gap-8">
                    <div className="flex items-center justify-between">
                        <h1 className="text-2xl font-semibold">Packaging Requests</h1>
                        <CreatePackagingRequestDialog onRequestCreated={handleRequestCreated}>
                            <Button>
                                <PlusCircle className="mr-2 h-4 w-4" />
                                Create New Request
                            </Button>
                        </CreatePackagingRequestDialog>
                    </div>
                    <Card>
                        <CardHeader>
                            <CardTitle>Request History</CardTitle>
                            <CardDescription>A log of all your packaging material requests.</CardDescription>
                        </CardHeader>
                        <CardContent>
                             <Table>
                                <TableHeader>
                                    <TableRow>
                                        <TableHead>Request ID</TableHead>
                                        <TableHead>Date</TableHead>
                                        <TableHead>Item</TableHead>
                                        <TableHead>Quantity</TableHead>
                                        <TableHead>Status</TableHead>
                                    </TableRow>
                                </TableHeader>
                                <TableBody>
                                    {requests.length === 0 ? (
                                        <TableRow>
                                            <TableCell colSpan={5} className="h-24 text-center">
                                                You haven't made any requests yet.
                                            </TableCell>
                                        </TableRow>
                                    ) : (
                                        requests.map((request) => (
                                            <TableRow key={request.id}>
                                                <TableCell className="font-medium">{request.id}</TableCell>
                                                <TableCell>{new Date(request.date).toLocaleDateString()}</TableCell>
                                                <TableCell>{request.item}</TableCell>
                                                <TableCell>{request.quantity}</TableCell>
                                                <TableCell>
                                                    <Badge variant="outline" className={statusStyles[request.status]}>
                                                        {request.status}
                                                    </Badge>
                                                </TableCell>
                                            </TableRow>
                                        ))
                                    )}
                                </TableBody>
                            </Table>
                        </CardContent>
                    </Card>
                </div>
            </DashboardLayout>
        </SidebarProvider>
    );
}
