
'use client';

import { useForm } from 'react-hook-form';
import { zodResolver } from '@hookform/resolvers/zod';
import { parcelSchema, type ParcelFormValues } from '@/lib/types';
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
  DialogDescription,
  DialogFooter,
  DialogClose,
  DialogTrigger,
} from '@/components/ui/dialog';
import {
  Form,
  FormControl,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from '@/components/ui/form';
import { Input } from '@/components/ui/input';
import { Button } from '@/components/ui/button';
import { Loader2 } from 'lucide-react';
import { ReactNode, useState } from 'react';

type UploadedOrder = Partial<ParcelFormValues> & { 
  id: string;
  errors?: Record<string, string>;
  isValid: boolean;
};

interface EditOrderDialogProps {
  order: UploadedOrder;
  onUpdate: (order: UploadedOrder) => void;
  children: ReactNode;
}

export function EditOrderDialog({ order, onUpdate, children }: EditOrderDialogProps) {
  const [open, setOpen] = useState(false);
  const form = useForm<ParcelFormValues>({
    resolver: zodResolver(parcelSchema),
    defaultValues: {
      orderType: order.orderType || 'Normal',
      orderReferenceNumber: order.orderReferenceNumber || '',
      orderAmount: order.orderAmount || 0,
      orderDate: order.orderDate || new Date().toISOString().split('T')[0],
      customerName: order.customerName || '',
      customerContact: order.customerContact || '',
      deliveryCity: order.deliveryCity || 'Anytown',
      deliveryAddress: order.deliveryAddress || '',
      airwayBillCopies: order.airwayBillCopies || 1,
      items: order.items || 1,
      pickupCity: order.pickupCity || 'MULTAN',
      pickupAddress: order.pickupAddress || 'Main Office',
      returnCity: order.returnCity || 'MULTAN',
      returnAddress: order.returnAddress || 'Main Office',
      bookingWeight: order.bookingWeight || 0.5,
      orderDetail: order.orderDetail || '',
      notes: order.notes || '',
      zone: order.zone || 'A',
    },
  });

  const { formState: { isSubmitting } } = form;

  const onSubmit = (values: ParcelFormValues) => {
    const updatedOrder: UploadedOrder = {
      ...order,
      ...values,
      isValid: true,
      errors: undefined,
    };
    onUpdate(updatedOrder);
    setOpen(false); // Close the dialog on successful submission
  };

  return (
    <Dialog open={open} onOpenChange={setOpen}>
      <DialogTrigger asChild>{children}</DialogTrigger>
      <DialogContent className="sm:max-w-xl">
        <DialogHeader>
          <DialogTitle>Edit Order</DialogTitle>
          <DialogDescription>
            Correct the errors for order reference: {order.orderReferenceNumber || 'N/A'}.
          </DialogDescription>
        </DialogHeader>
        <Form {...form}>
          <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 max-h-[60vh] overflow-y-auto p-1">
              {Object.keys(form.getValues()).map((key) => {
                const fieldKey = key as keyof ParcelFormValues;
                const errors = order.errors || {};
                const hasError = Object.keys(errors).includes(fieldKey);
                return (
                  <FormField
                    key={fieldKey}
                    control={form.control}
                    name={fieldKey}
                    render={({ field }) => (
                      <FormItem>
                        <FormLabel className={hasError ? 'text-red-500' : ''}>
                          {fieldKey.replace(/([A-Z])/g, ' $1').replace(/^./, str => str.toUpperCase())}
                        </FormLabel>
                        <FormControl>
                          <Input
                            placeholder={errors[fieldKey] || ''}
                            {...field}
                             onChange={e => field.onChange(
                                typeof form.getValues()[fieldKey] === 'number'
                                ? e.target.valueAsNumber
                                : e.target.value
                            )}
                          />
                        </FormControl>
                        <FormMessage />
                      </FormItem>
                    )}
                  />
                );
              })}
            </div>
            <DialogFooter>
              <DialogClose asChild>
                  <Button type="button" variant="outline">Cancel</Button>
              </DialogClose>
              <Button type="submit" disabled={isSubmitting}>
                {isSubmitting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
                Save Changes
              </Button>
            </DialogFooter>
          </form>
        </Form>
      </DialogContent>
    </Dialog>
  );
}
