
'use client';

import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import type { Complaint } from '@/lib/types';
import { Button } from '@/components/ui/button';
import { PlusCircle } from 'lucide-react';
import Link from 'next/link';
import { Avatar, AvatarFallback } from '../ui/avatar';
import { Badge } from '../ui/badge';
import { ScrollArea } from '../ui/scroll-area';

interface ComplaintsPreviewProps {
  complaints: Complaint[];
  onComplaintAdded: (complaint: Complaint) => void;
}

export function ComplaintsPreview({ complaints, onComplaintAdded }: ComplaintsPreviewProps) {
  const unreadCount = complaints.filter(c => c.status === 'Open' || c.status === 'In Progress').length;

  return (
    <Card className="h-full flex flex-col">
      <CardHeader className="flex flex-row items-center">
        <div className="grid gap-2">
            <CardTitle>My Complaints</CardTitle>
            <CardDescription>
                {unreadCount > 0 ? `You have ${unreadCount} open complaints.` : 'No open complaints.'}
            </CardDescription>
        </div>
        <Button asChild size="sm" className="ml-auto gap-1">
            <Link href="/new-complaint-request">
                <PlusCircle className="h-4 w-4" />
                Create Ticket
            </Link>
        </Button>
      </CardHeader>
      <CardContent className="p-0 flex-1">
        <ScrollArea className="h-full">
            <div className="flex flex-col">
            {complaints.length === 0 ? (
                <div className="flex flex-1 items-center justify-center py-10">
                    <p className="text-sm text-muted-foreground">You haven't created any tickets yet.</p>
                </div>
            ) : (
                complaints.map(complaint => (
                    <Link href="/complaints" key={complaint.id} className="block hover:bg-muted/50">
                        <div className="flex items-start gap-4 p-4">
                            <Avatar className="h-8 w-8">
                                <AvatarFallback className="text-xs">
                                    {complaint.status === 'Resolved' || complaint.status === 'Closed' ? '✓' : complaint.sellerId.charAt(0).toUpperCase()}
                                </AvatarFallback>
                            </Avatar>
                            <div className="grid gap-1 flex-1">
                                <div className="flex items-center justify-between">
                                    <p className={`font-semibold text-sm ${complaint.status !== 'Closed' && complaint.status !== 'Resolved' ? 'text-primary' : ''}`}>
                                        {complaint.subject}
                                    </p>
                                    {(complaint.status === 'Open' || complaint.status === 'In Progress') && (
                                        <Badge variant="destructive" className="h-5">1</Badge>
                                    )}
                                </div>
                                <p className="text-xs text-muted-foreground truncate">{complaint.description}</p>
                            </div>
                        </div>
                    </Link>
                ))
            )}
            </div>
        </ScrollArea>
      </CardContent>
    </Card>
  );
}
