'use client';

import { Bar, BarChart, ResponsiveContainer, XAxis, YAxis, Tooltip } from 'recharts';
import {
  Card,
  CardContent,
  CardHeader,
  CardTitle,
  CardDescription,
} from '@/components/ui/card';
import { ChartTooltipContent, ChartContainer } from '@/components/ui/chart';
import type { Parcel, ParcelStatus } from '@/lib/types';
import { useMemo } from 'react';

interface DeliveryPerformanceChartProps {
  parcels: Parcel[];
}

export function DeliveryPerformanceChart({ parcels }: DeliveryPerformanceChartProps) {
  const chartData = useMemo(() => {
    const statusCounts: Record<ParcelStatus, number> = {
      Delivered: 0,
      Pending: 0,
      'In Transit': 0,
      'Picked Up': 0,
      Returned: 0,
      Failed: 0,
    };

    parcels.forEach(parcel => {
      if (statusCounts[parcel.status] !== undefined) {
        statusCounts[parcel.status]++;
      }
    });

    return [
      { name: 'Delivered', count: statusCounts.Delivered, fill: 'var(--color-delivered)' },
      { name: 'Pending', count: statusCounts.Pending + statusCounts['Picked Up'] + statusCounts['In Transit'], fill: 'var(--color-pending)' },
      { name: 'Returned', count: statusCounts.Returned + statusCounts.Failed, fill: 'var(--color-returned)' },
    ];
  }, [parcels]);

  const chartConfig = {
    count: {
      label: 'Parcels',
    },
    delivered: {
      label: 'Delivered',
      color: 'hsl(var(--chart-1))',
    },
    pending: {
        label: 'Pending',
        color: 'hsl(var(--chart-4))',
    },
    returned: {
        label: 'Returned',
        color: 'hsl(var(--chart-2))',
    },
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle>Delivery Performance</CardTitle>
        <CardDescription>Overview of parcel statuses.</CardDescription>
      </CardHeader>
      <CardContent>
        <ChartContainer config={chartConfig} className="h-64 w-full">
          <ResponsiveContainer width="100%" height="100%">
            <BarChart data={chartData} margin={{ top: 5, right: 20, left: -10, bottom: 5 }}>
              <XAxis dataKey="name" tickLine={false} axisLine={false} tickMargin={8} fontSize={12} />
              <YAxis allowDecimals={false} tickLine={false} axisLine={false} tickMargin={8} fontSize={12} />
              <Tooltip
                cursor={false}
                content={<ChartTooltipContent indicator="dot" />}
              />
              <Bar dataKey="count" radius={8} />
            </BarChart>
          </ResponsiveContainer>
        </ChartContainer>
      </CardContent>
    </Card>
  );
}
