
'use client';

import * as React from 'react';
import Autoplay from 'embla-carousel-autoplay';
import { Card, CardContent } from '@/components/ui/card';
import {
  Carousel,
  CarouselContent,
  CarouselItem,
} from '@/components/ui/carousel';
import type { NewsItem } from '@/lib/types';
import { Badge } from '../ui/badge';
import { Newspaper } from 'lucide-react';
import Link from 'next/link';
import { formatDistanceToNow } from 'date-fns';

interface NewsSliderProps {
  newsItems: NewsItem[];
}

const categoryColors: Record<NewsItem['category'], string> = {
  Announcement: 'bg-blue-100 text-blue-800',
  Update: 'bg-green-100 text-green-800',
  Maintenance: 'bg-yellow-100 text-yellow-800',
};

export function NewsSlider({ newsItems }: NewsSliderProps) {
  const plugin = React.useRef(
    Autoplay({ delay: 5000, stopOnInteraction: true })
  );

  if (!newsItems || newsItems.length === 0) {
    return null;
  }

  return (
    <div className="relative">
        <Carousel
            plugins={[plugin.current]}
            className="w-full"
            onMouseEnter={plugin.current.stop}
            onMouseLeave={plugin.current.reset}
        >
            <CarouselContent>
            {newsItems.map((item) => (
                <CarouselItem key={item.id}>
                    <Link href="/company-news">
                        <Card className="bg-primary/5 border-primary/20 hover:border-primary/40 transition-colors">
                            <CardContent className="flex items-center gap-4 p-4">
                                <div className="hidden sm:flex h-10 w-10 items-center justify-center rounded-full bg-primary/10 text-primary">
                                    <Newspaper className="h-5 w-5" />
                                </div>
                                <div className="flex-1">
                                    <div className="flex flex-wrap items-center gap-x-4 gap-y-1">
                                        <Badge variant="outline" className={categoryColors[item.category]}>{item.category}</Badge>
                                        <h3 className="font-semibold text-sm sm:text-base">{item.title}</h3>
                                    </div>
                                    <p className="text-xs sm:text-sm text-muted-foreground mt-1">
                                        {item.content.substring(0, 100)}{item.content.length > 100 ? '...' : ''}
                                        <span className="ml-2 text-xs font-medium text-primary/80">{formatDistanceToNow(new Date(item.date), { addSuffix: true })}</span>
                                    </p>
                                </div>
                            </CardContent>
                        </Card>
                    </Link>
                </CarouselItem>
            ))}
            </CarouselContent>
        </Carousel>
    </div>
  );
}
