
'use client';

import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import {
  Wallet,
  Package,
  CheckCircle,
  RefreshCcw,
  Users,
  Truck,
  BookCopy,
  BookX,
  Warehouse,
  History,
  AlertTriangle,
  ArrowRightLeft,
  XCircle,
  PackageSearch,
  Building,
  Ban,
  Hammer,
} from 'lucide-react';
import type { Parcel, UserRole, ParcelStatus, TimeRange } from '@/lib/types';
import { useEffect, useState } from 'react';
import { subDays, isWithinInterval, startOfDay, endOfDay } from 'date-fns';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';

interface CardData {
  title: string;
  value: string | number;
  icon: React.ElementType;
  description?: string;
  color?: string;
  timeRange?: TimeRange;
  onTimeRangeChange?: (value: TimeRange) => void;
}

interface StatusCardData {
    label: string;
    value: number;
    icon: React.ElementType;
    color?: string;
}

const filterParcelsByDate = (parcels: Parcel[], days: number) => {
  if (days === 0) { // Special case for "Today"
      const todayStart = startOfDay(new Date());
      const todayEnd = endOfDay(new Date());
      return parcels.filter(p => {
        const bookingDate = new Date(p.bookingDate);
        return isWithinInterval(bookingDate, { start: todayStart, end: todayEnd });
      });
  }
  const endDate = new Date();
  const startDate = subDays(endDate, days);
  return parcels.filter(p => {
    const bookingDate = new Date(p.bookingDate);
    return isWithinInterval(bookingDate, { start: startDate, end: endDate });
  });
};

const countByStatus = (parcels: Parcel[], status: ParcelStatus | ParcelStatus[]) => {
  const statuses = Array.isArray(status) ? status : [status];
  return parcels.filter(p => statuses.includes(p.status)).length;
}

export function OverviewCards({ parcels, userRole = 'seller' }: { parcels: Parcel[]; userRole: UserRole }) {
  const [walletTime, setWalletTime] = useState<TimeRange>('30');
  const [deliveredTime, setDeliveredTime] = useState<TimeRange>('30');
  const [failedTime, setFailedTime] = useState<TimeRange>('30');
  const [statusTime, setStatusTime] = useState<TimeRange>('7');

  const [mainCards, setMainCards] = useState<CardData[]>([]);
  const [statusCards, setStatusCards] = useState<StatusCardData[]>([]);

  useEffect(() => {
    // Main Overview Cards
    const walletParcels = filterParcelsByDate(parcels, parseInt(walletTime));
    const deliveredParcels = filterParcelsByDate(parcels, parseInt(deliveredTime));
    const failedParcels = filterParcelsByDate(parcels, parseInt(failedTime));
    const todayParcels = filterParcelsByDate(parcels, 0);

    const walletAmount = walletParcels.reduce((sum, p) => sum + (p.status === 'Delivered' ? p.codAmount : 0), 0);
    
    // Status Overview Cards
    const statusParcels = filterParcelsByDate(parcels, parseInt(statusTime));

    setMainCards([
      {
        title: `PKR ${walletAmount.toLocaleString()}`,
        value: 'Wallet / COD Balance',
        icon: Wallet,
        timeRange: walletTime,
        onTimeRangeChange: (val) => setWalletTime(val),
      },
      {
        title: todayParcels.length.toString(),
        value: "Today's Booked",
        icon: Package,
      },
      {
        title: countByStatus(deliveredParcels, 'Delivered').toString(),
        value: 'Delivered',
        icon: CheckCircle,
        timeRange: deliveredTime,
        onTimeRangeChange: (val) => setDeliveredTime(val),
      },
      {
        title: countByStatus(failedParcels, ['Failed', 'Returned']).toString(),
        value: 'Return',
        icon: RefreshCcw,
        timeRange: failedTime,
        onTimeRangeChange: (val) => setFailedTime(val),
      },
    ]);
    
    setStatusCards([
        { label: 'Total Orders', value: statusParcels.length, icon: BookCopy, color: 'text-primary' },
        { label: 'Unbooked', value: countByStatus(statusParcels, 'Unbooked'), icon: BookX, color: 'text-muted-foreground' },
        { label: 'In Warehouse', value: countByStatus(statusParcels, 'In Warehouse'), icon: Warehouse, color: 'text-purple-500' },
        { label: 'Out for Delivery', value: countByStatus(statusParcels, 'Out for Delivery'), icon: Truck, color: 'text-cyan-500' },
        { label: 'Attempted', value: countByStatus(statusParcels, 'Attempted'), icon: History, color: 'text-yellow-500' },
        { label: 'Out for Return', value: countByStatus(statusParcels, 'Out for Return'), icon: RefreshCcw, color: 'text-orange-500' },
        { label: 'Returned', value: countByStatus(statusParcels, 'Returned'), icon: AlertTriangle, color: 'text-red-500' },
        { label: 'Shipper Advice', value: countByStatus(statusParcels, 'Shipper Advice'), icon: Users, color: 'text-indigo-500' },
        { label: 'Branch Shift', value: countByStatus(statusParcels, 'Branch Shift'), icon: Building, color: 'text-pink-500' },
        { label: 'Cancelled', value: countByStatus(statusParcels, 'Cancelled'), icon: Ban, color: 'text-gray-500' },
        { label: 'Lost', value: countByStatus(statusParcels, 'Lost'), icon: PackageSearch, color: 'text-red-600' },
        { label: 'Stolen', value: countByStatus(statusParcels, 'Stolen'), icon: XCircle, color: 'text-red-700' },
        { label: 'Damaged', value: countByStatus(statusParcels, 'Damaged'), icon: Hammer, color: 'text-red-800' },
    ]);
    
  }, [parcels, walletTime, deliveredTime, failedTime, statusTime]);

  if (userRole === 'admin') {
     return (
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
             <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">Total Bookings</CardTitle>
                    <BookCopy className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                    <div className="text-2xl font-bold">{parcels.length}</div>
                    <p className="text-xs text-muted-foreground">All-time parcel bookings</p>
                </CardContent>
            </Card>
             <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">Total Delivered</CardTitle>
                    <CheckCircle className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                    <div className="text-2xl font-bold">{parcels.filter(p=>p.status === 'Delivered').length}</div>
                    <p className="text-xs text-muted-foreground">All-time successful deliveries</p>
                </CardContent>
            </Card>
            <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">Total Sellers</CardTitle>
                    <Users className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                    <div className="text-2xl font-bold">125</div>
                    <p className="text-xs text-muted-foreground">Number of registered sellers</p>
                </CardContent>
            </Card>
            <Card>
                <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                    <CardTitle className="text-sm font-medium">Total Riders</CardTitle>
                    <Truck className="h-4 w-4 text-muted-foreground" />
                </CardHeader>
                <CardContent>
                    <div className="text-2xl font-bold">42</div>
                    <p className="text-xs text-muted-foreground">Number of active riders</p>
                </CardContent>
            </Card>
        </div>
    )
  }

  return (
    <div className="flex flex-col gap-6">
        <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            {mainCards.map((card) => (
            <Card key={card.value}>
                <CardHeader className="pb-2">
                    <div className="flex items-center justify-between">
                        <CardTitle className="text-sm font-medium">{card.value}</CardTitle>
                        {card.timeRange && card.onTimeRangeChange && (
                            <Select value={card.timeRange} onValueChange={card.onTimeRangeChange}>
                                <SelectTrigger className="h-7 w-[90px] text-xs [&_span]:truncate">
                                    <SelectValue />
                                </SelectTrigger>
                                <SelectContent>
                                    <SelectItem value="7">Last 7d</SelectItem>
                                    <SelectItem value="15">Last 15d</SelectItem>
                                    <SelectItem value="30">Last 30d</SelectItem>
                                </SelectContent>
                            </Select>
                        )}
                    </div>
                </CardHeader>
                <CardContent>
                    <div className="flex items-center gap-2">
                        <card.icon className="h-6 w-6 text-muted-foreground" />
                        <div className="text-2xl font-bold">{card.title}</div>
                    </div>
                </CardContent>
            </Card>
            ))}
        </div>

        <Card>
            <CardHeader>
                <div className="flex items-center justify-between">
                    <CardTitle>Order Status Overview</CardTitle>
                    <Select value={statusTime} onValueChange={(val) => setStatusTime(val as TimeRange)}>
                        <SelectTrigger className="h-8 w-[120px] text-sm">
                            <SelectValue />
                        </SelectTrigger>
                        <SelectContent>
                            <SelectItem value="0">Today</SelectItem>
                            <SelectItem value="7">Last 7 days</SelectItem>
                            <SelectItem value="15">Last 15 days</SelectItem>
                            <SelectItem value="30">Last 30 days</SelectItem>
                        </SelectContent>
                    </Select>
                </div>
            </CardHeader>
            <CardContent>
                <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-7 gap-4">
                    {statusCards.map(card => (
                         <Card key={card.label} className="flex flex-col items-center justify-center p-4 text-center">
                            <card.icon className={`h-6 w-6 mb-2 ${card.color || 'text-muted-foreground'}`} />
                            <p className="text-xl font-bold">{card.value}</p>
                            <p className="text-xs text-muted-foreground">{card.label}</p>
                        </Card>
                    ))}
                </div>
            </CardContent>
        </Card>

    </div>
  );
}
