
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from '@/components/ui/table';
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { ArrowUpRight } from 'lucide-react';
import type { Parcel, ParcelStatus } from '@/lib/types';
import { format } from 'date-fns';
import Link from 'next/link';

const statusStyles: Record<ParcelStatus, string> = {
  Pending: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-300',
  'Picked Up': 'bg-blue-100 text-blue-800 dark:bg-blue-900/50 dark:text-blue-300',
  'In Transit': 'bg-cyan-100 text-cyan-800 dark:bg-cyan-900/50 dark:text-cyan-300',
  Delivered: 'bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-300',
  Returned: 'bg-orange-100 text-orange-800 dark:bg-orange-900/50 dark:text-orange-300',
  Failed: 'bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-300',
  Unbooked: 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300',
  'In Warehouse': 'bg-purple-100 text-purple-800 dark:bg-purple-900/50 dark:text-purple-300',
  'Out for Delivery': 'bg-blue-100 text-blue-800 dark:bg-blue-900/50 dark:text-blue-300',
  Attempted: 'bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-300',
  'Out for Return': 'bg-orange-100 text-orange-800 dark:bg-orange-900/50 dark:text-orange-300',
  'Shipper Advice': 'bg-indigo-100 text-indigo-800 dark:bg-indigo-900/50 dark:text-indigo-300',
  'Branch Shift': 'bg-pink-100 text-pink-800 dark:bg-pink-900/50 dark:text-pink-300',
  Cancelled: 'bg-gray-200 text-gray-800 dark:bg-gray-800 dark:text-gray-200',
  Lost: 'bg-red-200 text-red-900 dark:bg-red-900/50 dark:text-red-200',
  Stolen: 'bg-red-300 text-red-900 dark:bg-red-800/50 dark:text-red-200',
  Damaged: 'bg-red-200 text-red-900 dark:bg-red-900/60 dark:text-red-200',
};

export function ParcelsTable({ 
    parcels,
    title = "Recent Parcels",
    description = "Your most recent parcel bookings.",
    viewAllLink = "/parcels",
    showViewAll = true,
    showCardWrapper = true,
 }: { 
    parcels: Parcel[],
    title?: string,
    description?: string,
    viewAllLink?: string,
    showViewAll?: boolean,
    showCardWrapper?: boolean
 }) {
  const TableContent = () => (
     <div className="overflow-x-auto">
        <Table>
            <TableHeader>
            <TableRow>
                <TableHead>Order ID</TableHead>
                <TableHead>Customer</TableHead>
                <TableHead>City</TableHead>
                <TableHead className="text-right">Amount</TableHead>
                <TableHead>Status</TableHead>
                <TableHead>Date</TableHead>
            </TableRow>
            </TableHeader>
            <TableBody>
            {parcels.length === 0 ? (
                <TableRow>
                <TableCell colSpan={6} className="h-24 text-center">
                    No recent orders found.
                </TableCell>
                </TableRow>
            ) : parcels.map(parcel => (
                <TableRow key={parcel.id}>
                <TableCell className="font-medium">{parcel.id}</TableCell>
                <TableCell>{parcel.recipientName}</TableCell>
                <TableCell>{parcel.city}</TableCell>
                <TableCell className="text-right">PKR {parcel.codAmount.toFixed(2)}</TableCell>
                <TableCell>
                    <Badge variant="outline" className={`font-semibold ${statusStyles[parcel.status]}`}>{parcel.status}</Badge>
                </TableCell>
                <TableCell>{format(new Date(parcel.bookingDate), 'dd MMM')}</TableCell>
                </TableRow>
            ))}
            </TableBody>
        </Table>
    </div>
  )

  if (!showCardWrapper) {
    return <TableContent />;
  }

  return (
    <Card className="h-full">
      <CardHeader className="flex flex-row items-center">
        <div className="grid gap-2">
            <CardTitle>{title}</CardTitle>
            <CardDescription>{description}</CardDescription>
        </div>
        {showViewAll && (
            <Button asChild size="sm" className="ml-auto gap-1">
                <Link href={viewAllLink}>
                    View All
                    <ArrowUpRight className="h-4 w-4" />
                </Link>
            </Button>
        )}
      </CardHeader>
      <CardContent className="p-0">
        <TableContent />
      </CardContent>
    </Card>
  );
}
