
'use client';

import { useState, useEffect, useCallback } from 'react';
import { Input } from '@/components/ui/input';
import { Search, Package, Users, Building } from 'lucide-react';
import { Popover, PopoverContent, PopoverTrigger } from '@/components/ui/popover';
import { Command, CommandEmpty, CommandGroup, CommandInput, CommandItem, CommandList } from '@/components/ui/command';
import { mockParcels, mockSellers, mockBranches } from '@/lib/data';
import Link from 'next/link';
import { useDebounce } from '@/hooks/use-debounce';

type SearchResult = {
    type: 'Shipment' | 'Merchant' | 'Branch';
    id: string;
    label: string;
    description: string;
    href: string;
    icon: React.ElementType;
};

const allData = [
    ...mockParcels.map(p => ({ type: 'Shipment' as const, id: p.id, label: p.id, description: `To: ${p.recipientName}, ${p.city}`, href: `/admin/shipments`, icon: Package })),
    ...mockSellers.map(s => ({ type: 'Merchant' as const, id: s.id, label: s.name, description: s.email, href: '/admin/merchants', icon: Users })),
    ...mockBranches.map(b => ({ type: 'Branch' as const, id: b.id, label: b.name, description: b.location, href: '/admin/branches', icon: Building })),
];

export function GlobalSearch() {
  const [open, setOpen] = useState(false);
  const [query, setQuery] = useState('');
  const debouncedQuery = useDebounce(query, 300);
  const [results, setResults] = useState<SearchResult[]>([]);

  useEffect(() => {
    if (debouncedQuery.length > 1) {
        const filteredResults = allData.filter(item =>
            item.label.toLowerCase().includes(debouncedQuery.toLowerCase()) ||
            item.description.toLowerCase().includes(debouncedQuery.toLowerCase())
        );
        setResults(filteredResults);
        if (filteredResults.length > 0) {
            setOpen(true);
        } else {
            setOpen(false);
        }
    } else {
        setResults([]);
        setOpen(false);
    }
  }, [debouncedQuery]);

  const handleSelect = useCallback(() => {
    setQuery('');
    setResults([]);
    setOpen(false);
  }, []);

  return (
    <Popover open={open} onOpenChange={setOpen}>
        <PopoverTrigger asChild>
            <div className="relative hidden md:block w-full max-w-sm">
                <Search className="absolute left-3 top-1/2 -translate-y-1/2 h-4 w-4 text-muted-foreground" />
                <Input 
                    value={query}
                    onChange={(e) => setQuery(e.target.value)}
                    placeholder="Search shipments, merchants, branches..." 
                    className="h-9 w-full rounded-md border border-input bg-transparent pl-9 text-sm" 
                />
            </div>
        </PopoverTrigger>
        <PopoverContent className="w-[var(--radix-popover-trigger-width)] p-0" align="start">
             <Command>
                <CommandInput placeholder="Type to search..." value={query} onValueChange={setQuery} />
                <CommandList>
                    <CommandEmpty>No results found.</CommandEmpty>
                    <CommandGroup heading="Suggestions">
                        {results.map((result) => (
                            <Link href={result.href} key={`${result.type}-${result.id}`} passHref>
                                <CommandItem onSelect={handleSelect} value={`${result.label} ${result.description}`}>
                                    <result.icon className="mr-2 h-4 w-4" />
                                    <span>{result.label}</span>
                                    <span className="ml-4 text-xs text-muted-foreground">{result.description}</span>
                                </CommandItem>
                            </Link>
                        ))}
                    </CommandGroup>
                </CommandList>
            </Command>
        </PopoverContent>
    </Popover>
  );
}
