
'use client';

import { useState, type ReactNode, useMemo } from 'react';
import { useToast } from '@/hooks/use-toast';
import type { PackagingRequest, PackagingItem } from '@/lib/types';

import { Button } from '@/components/ui/button';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogFooter,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
  DialogClose,
} from '@/components/ui/dialog';
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from '@/components/ui/select';
import { Input } from '@/components/ui/input';
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow, TableFooter as ShadTableFooter } from '@/components/ui/table';
import { Loader2, Trash2 } from 'lucide-react';

interface CreatePackagingRequestDialogProps {
  children: ReactNode;
  onRequestCreated: (request: PackagingRequest) => void;
}

const serviceTypes: PackagingItem[] = [
    { name: 'Small Flyers 10x12', price: 10 },
    { name: 'Medium 12x16', price: 15 },
    { name: 'Large 14x19', price: 20 },
    { name: 'Miscellaneous', price: 5 },
    { name: 'XL Flyer 18x21', price: 25 },
    { name: 'Fragile Sticker', price: 2 },
    { name: 'Printed Box 1 KG (10 x 8 x 3)', price: 50 },
    { name: 'Printed Box 2 KG (13.5 x 11 x 3)', price: 70 },
    { name: 'Printed Box 5 KG (12 x 10.5 x 8.5)', price: 100 },
    { name: 'Printed Box 10 KG (16 x 12 x 10)', price: 150 },
    { name: 'Printed Box 15 KG (19 x 15 x 15)', price: 200 },
];

type RequestListItem = {
    item: PackagingItem;
    quantity: number;
};

export function CreatePackagingRequestDialog({ children, onRequestCreated }: CreatePackagingRequestDialogProps) {
  const [open, setOpen] = useState(false);
  const [isSubmitting, setIsSubmitting] = useState(false);
  const { toast } = useToast();
  
  const [city, setCity] = useState('');
  const [address, setAddress] = useState('');
  const [serviceType, setServiceType] = useState<PackagingItem | null>(null);
  const [quantity, setQuantity] = useState<number | string>('');
  const [requestList, setRequestList] = useState<RequestListItem[]>([]);

  const handleAddRequest = () => {
    if (serviceType && quantity) {
        const numQuantity = Number(quantity);
        if (numQuantity > 0) {
            setRequestList(prev => [...prev, { item: serviceType, quantity: numQuantity }]);
            setServiceType(null);
            setQuantity('');
        }
    }
  };

  const handleRemoveRequest = (index: number) => {
    setRequestList(prev => prev.filter((_, i) => i !== index));
  };
  
  const grandTotal = useMemo(() => {
    return requestList.reduce((total, req) => total + (req.item.price * req.quantity), 0);
  }, [requestList]);

  const handleSubmit = async () => {
    if (requestList.length === 0) {
        toast({ variant: 'destructive', title: 'Error', description: 'Please add at least one item to the request.' });
        return;
    }
    setIsSubmitting(true);
    // Mock API call
    await new Promise(resolve => setTimeout(resolve, 1000));

    // In a real app, you might create one request per item or a single request with multiple items
    const firstItem = requestList[0];
    const newRequest: PackagingRequest = {
        id: `REQ-${Date.now().toString().slice(-6)}`,
        date: new Date().toISOString(),
        item: `${firstItem.item.name} ${requestList.length > 1 ? ` & ${requestList.length - 1} more` : ''}`,
        quantity: requestList.reduce((sum, i) => sum + i.quantity, 0),
        status: 'Pending',
    };
    
    onRequestCreated(newRequest);
    resetAndClose();
    toast({
        title: 'Request Submitted!',
        description: `Your packaging request has been submitted successfully.`,
    });
  };

  const resetAndClose = () => {
    setCity('');
    setAddress('');
    setServiceType(null);
    setQuantity('');
    setRequestList([]);
    setIsSubmitting(false);
    setOpen(false);
  }

  return (
    <Dialog open={open} onOpenChange={(isOpen) => { if (!isOpen) resetAndClose(); else setOpen(true); }}>
      <DialogTrigger asChild>
        {children}
      </DialogTrigger>
      <DialogContent className="sm:max-w-4xl">
        <DialogHeader>
          <DialogTitle>Create Request</DialogTitle>
          <DialogDescription>
            Request new packaging materials for your shipments.
          </DialogDescription>
        </DialogHeader>
        <div className="grid gap-6 py-4">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label className="text-sm font-medium">Merchant City</label>
                    <Select value={city} onValueChange={setCity}>
                        <SelectTrigger><SelectValue placeholder="Select a city" /></SelectTrigger>
                        <SelectContent>
                            <SelectItem value="multan">Multan</SelectItem>
                            <SelectItem value="lahore">Lahore</SelectItem>
                            <SelectItem value="karachi">Karachi</SelectItem>
                        </SelectContent>
                    </Select>
                </div>
                 <div>
                    <label className="text-sm font-medium">Address</label>
                    <Select value={address} onValueChange={setAddress}>
                        <SelectTrigger><SelectValue placeholder="Select an address" /></SelectTrigger>
                        <SelectContent>
                            <SelectItem value="main-office">Main Office</SelectItem>
                            <SelectItem value="warehouse-a">Warehouse A</SelectItem>
                        </SelectContent>
                    </Select>
                </div>
                 <div className="md:col-span-2 grid grid-cols-1 sm:grid-cols-[2fr_1fr_auto] gap-4 items-end">
                    <div>
                        <label className="text-sm font-medium">Service type</label>
                        <Select
                            value={serviceType?.name}
                            onValueChange={(val) => setServiceType(serviceTypes.find(s => s.name === val) || null)}
                        >
                            <SelectTrigger><SelectValue placeholder="Select a service type" /></SelectTrigger>
                            <SelectContent>
                                {serviceTypes.map(st => <SelectItem key={st.name} value={st.name}>{st.name}</SelectItem>)}
                            </SelectContent>
                        </Select>
                    </div>
                     <div>
                        <label className="text-sm font-medium">Quantity</label>
                        <Input 
                            type="number" 
                            placeholder="Enter Quantity" 
                            value={quantity} 
                            onChange={(e) => setQuantity(e.target.value)} 
                        />
                    </div>
                    <Button onClick={handleAddRequest} disabled={!serviceType || !quantity}>Add Request</Button>
                </div>
            </div>

            {requestList.length > 0 && (
                <div className="max-h-64 overflow-y-auto border rounded-lg">
                    <Table>
                        <TableHeader>
                            <TableRow>
                                <TableHead>Package</TableHead>
                                <TableHead className="text-right">Price</TableHead>
                                <TableHead className="text-right">Quantity</TableHead>
                                <TableHead className="text-right">Amount</TableHead>
                                <TableHead className="w-[80px]">Actions</TableHead>
                            </TableRow>
                        </TableHeader>
                        <TableBody>
                            {requestList.map((req, index) => (
                                <TableRow key={index}>
                                    <TableCell>{req.item.name}</TableCell>
                                    <TableCell className="text-right">PKR {req.item.price.toFixed(2)}</TableCell>
                                    <TableCell className="text-right">{req.quantity}</TableCell>
                                     <TableCell className="text-right font-medium">PKR {(req.item.price * req.quantity).toFixed(2)}</TableCell>
                                    <TableCell className="text-center">
                                        <Button variant="ghost" size="icon" className="h-8 w-8" onClick={() => handleRemoveRequest(index)}>
                                            <Trash2 className="h-4 w-4 text-red-500" />
                                        </Button>
                                    </TableCell>
                                </TableRow>
                            ))}
                        </TableBody>
                         <ShadTableFooter>
                            <TableRow>
                                <TableCell colSpan={3} className="text-right font-bold">Total Amount</TableCell>
                                <TableCell className="text-right font-bold">PKR {grandTotal.toFixed(2)}</TableCell>
                                <TableCell />
                            </TableRow>
                        </ShadTableFooter>
                    </Table>
                </div>
            )}
        </div>
        <DialogFooter className="mt-4">
          <DialogClose asChild>
            <Button type="button" variant="destructive">
              Close
            </Button>
          </DialogClose>
          <Button type="button" onClick={handleSubmit} disabled={isSubmitting || requestList.length === 0}>
            {isSubmitting && <Loader2 className="mr-2 h-4 w-4 animate-spin" />}
            {isSubmitting ? 'Submitting Request...' : 'Submit Request'}
          </Button>
        </DialogFooter>
      </DialogContent>
    </Dialog>
  );
}
