
'use client';

import {
  AlertDialog,
  AlertDialogAction,
  AlertDialogContent,
  AlertDialogDescription,
  AlertDialogFooter,
  AlertDialogHeader,
  AlertDialogTitle,
} from '@/components/ui/alert-dialog';
import { Badge } from '@/components/ui/badge';
import type { Parcel } from '@/lib/types';
import { format } from 'date-fns';
import { Clock, HelpCircle, PackageCheck, XCircle, Search, Loader2 } from 'lucide-react';
import { Dialog, DialogTrigger, DialogContent, DialogHeader, DialogTitle, DialogDescription, DialogFooter } from '@/components/ui/dialog';
import { Button } from './ui/button';
import { Input } from './ui/input';
import { ReactNode } from 'react';

interface TrackParcelDialogProps {
  parcel?: Parcel | null;
  open: boolean;
  onOpenChange: (open: boolean) => void;
  trackingId: string;
  isTracking: boolean;
}

interface NewTrackParcelDialogProps {
    children: ReactNode;
    onTrack: (id: string) => void;
    isTracking: boolean;
    trackingId: string;
    onIdChange: (id: string) => void;
}

export function TrackParcelDialog({ parcel, open, onOpenChange, trackingId, isTracking }: TrackParcelDialogProps) {
  const getStatusInfo = (status: Parcel['status'] | undefined) => {
    switch (status) {
      case 'Delivered':
        return { icon: PackageCheck, color: 'text-green-500', text: 'Delivered' };
      case 'In Transit':
      case 'Picked Up':
        return { icon: Clock, color: 'text-blue-500', text: 'In Transit' };
      case 'Pending':
        return { icon: Clock, color: 'text-yellow-500', text: 'Pending' };
      case 'Failed':
      case 'Returned':
        return { icon: XCircle, color: 'text-red-500', text: 'Failed/Returned' };
      default:
        return { icon: HelpCircle, color: 'text-muted-foreground', text: 'Not Found' };
    }
  };

  const statusInfo = getStatusInfo(parcel?.status);

  return (
    <AlertDialog open={open} onOpenChange={onOpenChange}>
      <AlertDialogContent>
        <AlertDialogHeader>
          <AlertDialogTitle className="flex items-center gap-2 text-xl">
             {isTracking ? <Loader2 className="h-6 w-6 animate-spin" /> : <statusInfo.icon className={`h-6 w-6 ${statusInfo.color}`} />}
            Tracking Status
          </AlertDialogTitle>
          <AlertDialogDescription className="pt-2">
            Result for tracking ID: <Badge variant="secondary" className="font-mono">{trackingId}</Badge>
          </AlertDialogDescription>
        </AlertDialogHeader>

        <div className="space-y-4 py-4 min-h-[150px]">
          {isTracking ? (
             <div className="flex items-center justify-center h-full">
                <Loader2 className="h-8 w-8 animate-spin text-muted-foreground" />
             </div>
          ) : parcel ? (
            <div className="rounded-lg border bg-card-foreground/5 p-4 space-y-4">
              <div className="grid grid-cols-2 gap-4">
                <div>
                  <p className="text-sm text-muted-foreground">Recipient</p>
                  <p className="font-semibold text-foreground">{parcel.recipientName}</p>
                </div>
                <div>
                  <p className="text-sm text-muted-foreground">Status</p>
                  <p className={`font-semibold ${statusInfo.color}`}>{parcel.status}</p>
                </div>
              </div>
              <div>
                <p className="text-sm text-muted-foreground">Booking Date</p>
                <p className="font-semibold text-foreground">{format(new Date(parcel.bookingDate), "eee, MMM d, yyyy")}</p>
              </div>
               <div>
                <p className="text-sm text-muted-foreground">Last Update</p>
                <p className="text-sm text-foreground">
                  {parcel.status === 'Delivered'
                    ? `Your parcel was delivered on ${format(new Date(), "MMM d, yyyy")}.`
                    : parcel.status === 'In Transit'
                    ? 'Your parcel is on its way to the destination city.'
                    : 'The courier has picked up the parcel from the seller.'}
                </p>
              </div>
            </div>
          ) : (
            <div className="text-center py-8">
              <p className="text-muted-foreground">No parcel found with this tracking ID.</p>
              <p className="text-sm text-muted-foreground">Please check the ID and try again.</p>
            </div>
          )}
        </div>

        <AlertDialogFooter>
          <AlertDialogAction onClick={() => onOpenChange(false)}>
            Close
          </AlertDialogAction>
        </AlertDialogFooter>
      </AlertDialogContent>
    </AlertDialog>
  );
}

export function NewTrackParcelDialog({ children, onTrack, isTracking, trackingId, onIdChange }: NewTrackParcelDialogProps) {
    return (
        <Dialog>
            <DialogTrigger asChild>
                {children}
            </DialogTrigger>
            <DialogContent>
                <DialogHeader>
                    <DialogTitle>Track Your Shipment</DialogTitle>
                    <DialogDescription>Enter your tracking ID below to see the latest status of your shipment.</DialogDescription>
                </DialogHeader>
                <div className="flex items-center space-x-2 py-4">
                    <Input 
                        placeholder="Enter Tracking ID" 
                        value={trackingId}
                        onChange={(e) => onIdChange(e.target.value)}
                    />
                    <Button onClick={() => onTrack(trackingId)} disabled={isTracking || !trackingId}>
                        {isTracking ? <Loader2 className="h-4 w-4 animate-spin" /> : <Search className="h-4 w-4" />}
                    </Button>
                </div>
                <DialogFooter>
                    <p className="text-xs text-muted-foreground">
                        You can find the tracking ID in your order confirmation email or on your dashboard.
                    </p>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    )
}
