

import type { PredictDeliveryETAOutput } from "@/ai/flows/delivery-eta-prediction";
import { z } from 'zod';

export const parcelSchema = z.object({
  orderType: z.string().min(1, { message: "Order type is required." }),
  orderReferenceNumber: z.string().min(3, { message: "Order reference number must be at least 3 characters." }),
  orderAmount: z.coerce.number().min(0, { message: "Order amount cannot be negative." }),
  orderDate: z.string().min(1, { message: "Order date is required." }),
  customerName: z.string().min(3, { message: "Customer name must be at least 3 characters." }),
  customerContact: z.string().min(10, { message: "Contact number must be at least 10 digits." }),
  deliveryCity: z.string().min(1, { message: "Delivery city is required." }),
  deliveryAddress: z.string().min(10, { message: "Delivery address must be at least 10 characters." }),
  airwayBillCopies: z.coerce.number().min(1, { message: "At least 1 copy is required." }),
  items: z.coerce.number().min(1, { message: "At least 1 item is required." }),
  pickupCity: z.string().min(1, { message: "Pickup city is required." }),
  pickupAddress: z.string().min(1, { message: "Pickup address is required." }),
  returnCity: z.string().min(1, { message: "Return city is required." }),
  returnAddress: z.string().min(1, { message: "Return address is required." }),
  bookingWeight: z.coerce.number().min(0.1, { message: "Booking weight must be at least 0.1 kg." }),
  orderDetail: z.string().optional(),
  notes: z.string().optional(),
  zone: z.string().min(1, { message: "Zone is required." }),
});

export const complaintSchema = z.object({
  parcelId: z.string().min(1, { message: "Parcel ID is required." }),
  subject: z.string().min(5, { message: "Subject must be at least 5 characters." }),
  description: z.string().min(20, { message: "Description must be at least 20 characters." }),
});

export type ComplaintFormValues = z.infer<typeof complaintSchema>;


export type ParcelFormValues = z.infer<typeof parcelSchema>;

export type ParcelStatus = 
  | 'Pending' 
  | 'Picked Up' 
  | 'In Transit' 
  | 'Delivered' 
  | 'Returned' 
  | 'Failed'
  | 'Unbooked'
  | 'In Warehouse'
  | 'Out for Delivery'
  | 'Attempted'
  | 'Out for Return'
  | 'Shipper Advice'
  | 'Branch Shift'
  | 'Cancelled'
  | 'Lost'
  | 'Stolen'
  | 'Damaged';


export type Parcel = {
  id: string;
  recipientName: string; // Will be mapped from customerName
  address: string; // Will be mapped from deliveryAddress
  contact: string; // Will be mapped from customerContact
  codAmount: number; // Will be mapped from orderAmount
  city: string; // Will be mapped from deliveryCity
  zone: string;
  status: ParcelStatus;
  bookingDate: string; // Will be mapped from orderDate
  assignedRiderId?: string;
  sellerId: string;
  isScanned?: boolean; // To control airway bill printing

  // New fields
  orderType?: string;
  orderReferenceNumber?: string;
  items?: number;
  bookingWeight?: number;
  pickupCity?: string;
  pickupAddress?: string;
  returnCity?: string;
  returnAddress?: string;
  orderDetail?: string;
  notes?: string;
};

export type Rider = {
  id:string;
  name: string;
  phone: string;
  zone: string;
  avatar: string;
};

export type Seller = {
  id: string;
  name: string;
  companyName: string;
  email: string;
  avatar: string;
};

export type Branch = {
  id: string;
  name: string;
  manager: string;
  location: string;
  totalShipments: number;
  avatar: string;
};

export type UserRole = 'seller' | 'admin' | 'super-admin' | 'finance' | 'branch-operator' | 'marketing' | 'support' | 'rider-manager' | 'finance-manager' | 'marketing-manager' | 'hr';

export type User = Seller & {
  role: UserRole;
};

export type BookingResult = {
  parcel: Parcel;
  eta: PredictDeliveryETAOutput;
} | null;

export type Payout = {
  id: string;
  date: string;
  amount: number;
  status: 'Pending' | 'Paid' | 'Failed';
  transactionId?: string;
}

export type TimeRange = '0' | '7' | '30' | '365' | '-1' | 'custom';

export type ComplaintStatus = 'Open' | 'In Progress' | 'Resolved' | 'Closed';

export type Complaint = {
  id: string;
  parcelId: string;
  subject: string;
  description: string;
  status: ComplaintStatus;
  date: string;
  sellerId: string;
};

export type PackagingRequest = {
    id: string;
    date: string;
    item: string;
    quantity: number;
    status: 'Pending' | 'Shipped' | 'Delivered';
};

export type PackagingLog = {
    id: string;
    packageNumber: string;
    price: number;
    merchantName: string;
    address: string;
    city: string;
    packageDate: string;
    description: string;
    status: 'Pending' | 'Approved' | 'Un-Assigned';
}

export type PackagingItem = {
    name: string;
    price: number;
}

export type NewsItem = {
  id: string;
  title: string;
  content: string;
  date: string;
  category: 'Update' | 'Announcement' | 'Maintenance';
};
